<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */
namespace app\modules\subscription\services;

use Stripe\Customer;
use Stripe\Event as StripeEvent;
use Stripe\PaymentIntent;
use Stripe\StripeClient as Client;
use Yii;
use yii\helpers\Url;

/**
 * Class StripeClient
 * @package app\modules\subscription\services
 */
class StripeClient
{

    public $client;

    /**
     * StripeClient constructor.
     */
    public function __construct()
    {
        $settings = Yii::$app->settings;

        // Stripe env
        $environment = $settings->get('subscription.environment');

        // Stripe keys
        $publishableKey = $settings->get('subscription.stripeLivePublishableKey');
        $secretKey = $settings->get('subscription.stripeLiveSecretKey');

        if ($environment === 'sandbox') {
            $publishableKey = $settings->get('subscription.stripeSandboxPublishableKey');
            $secretKey = $settings->get('subscription.stripeSandboxSecretKey');
        }

        $this->client = new Client($secretKey);
        \Stripe\Stripe::setApiKey($secretKey);

    }

    /**
     * Create customer
     *
     * @param array $params
     * @return Customer
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function customer(array $params)
    {
        return $this->client->customers->create($params);
    }

    /**
     * API call to find a customer
     *
     * Params Eg.
     * "email" => "john.doe@example.com",
     *
     * @param array $params
     * @return string
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function findCustomerIdBy(array $params)
    {
        $params = array_merge(['limit' => 1], $params);
        $customers = $this->client->customers->all($params);
        if (!isset($customers['data'], $customers['data'][0]) && !empty($customers['data'][0]->id)) {
            return $customers['data'][0]->id;
        }

        return '';
    }

    /**
     * Create Checkout Session
     *
     * @param $params
     * @return \Stripe\Checkout\Session
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function session($params)
    {
        return \Stripe\Checkout\Session::create($params);
    }

    /**
     * Retrieve Checkout Session By ID
     *
     * @param $id
     * @return \Stripe\Checkout\Session
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveSessionByById($id)
    {
        return \Stripe\Checkout\Session::retrieve($id);
    }

    /**
     * Retrieve Payment Intent By ID
     *
     * @param $id
     * @return \Stripe\PaymentIntent
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrievePaymentIntent($id)
    {
        return $this->client->paymentIntents->retrieve($id);
    }

    /**
     * Retrieve Plan By ID
     *
     * @param $plan_id
     * @return \Stripe\Plan
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrievePlanByID($plan_id)
    {
        return $this->client->plans->retrieve($plan_id);
    }

    /**
     * Create Subscription
     *
     * @param $customer_id
     * @param $items
     * @return \Stripe\Subscription
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function createSubscription($customer_id, $items)
    {
        return $this->client->subscriptions->create([
            'customer' => $customer_id,
            'items' => $items,
        ]);
    }

    /**
     * Get Stripe Subscription
     *
     * @param $subscription_id
     * @return \Stripe\Subscription
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function getSubscription($subscription_id)
    {
        return $this->client->subscriptions->retrieve($subscription_id, []);
    }

    /**
     * Cancel Stripe Subscription
     *
     * @param $subscription_id
     * @return \Stripe\Subscription
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function cancelSubscription($subscription_id)
    {
        return $this->client->subscriptions->cancel($subscription_id, []);
    }

    /**
     * Get Stripe Payment Intent
     *
     * @param $amount
     * @param string $currency_code
     * @return PaymentIntent
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function paymentIntent($amount, $currency_code = 'usd')
    {
        return $this->client->paymentIntents->create([
            'amount' => $amount,
            'currency' => $currency_code,
            // Verify your integration in this guide by including this parameter
            'metadata' => ['integration_check' => 'accept_a_payment'],
        ]);
    }

    /**
     * Confirm Stripe Payment Intent
     *
     * @param $payment_intent_id
     * @return PaymentIntent
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function confirmPaymentIntent($payment_intent_id)
    {
        $intent = $this->client->paymentIntents->retrieve($payment_intent_id);
        // You cannot confirm this PaymentIntent
        // because it has already succeeded after being previously confirmed.
        if (isset($intent->status) && $intent->status !== 'succeeded') {
            return $intent->confirm();
        }
        return $intent;
    }

    /**
     * Retrieve Stripe Event By ID
     *
     * @param $event_id
     * @return StripeEvent
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveEventByID($event_id)
    {
        return $this->client->events->retrieve($event_id);
    }

    public function collectPaymentMethod($subscription_id, $new_price_id, $return_url)
    {
        $subscription = $this->getSubscription($subscription_id);

        $successUrl = Url::to([
            '/subscription/user/update-payment-method',
            'id' => $subscription_id,
            'price_id' => $subscription->items->data[0]->id,
            'now' => $new_price_id,
            'return_url' => $return_url,
        ], true);
        $successUrl = $successUrl . "&ses_id={CHECKOUT_SESSION_ID}";


        $session = $this->client->checkout->sessions->create([
            'currency' => 'usd',
            'mode' => 'setup',
            'success_url' => $successUrl,
            'cancel_url' => $return_url,
//            'cancel_url' => Url::to([
//                '/subscription/user/confirm',
//                'payment' => 'canceled',
//                'id' => $subscription_id
//            ], true),
        ]);
        return $session->url;
    }

    /**
     * Update Stripe Subscription
     *
     * @param $subscription_id
     * @param $new_price_id
     * @return \Stripe\Subscription|null
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function updateSubscription($subscription_id, $new_price_id)
    {
        $subscription = $this->getSubscription($subscription_id);

        if (isset($subscription->items->data[0]->id)) {
            return $this->client->subscriptions->update(
                $subscription_id, [
                    'items' => [[
                        'id' => $subscription->items->data[0]->id,
                        'price' => $new_price_id,
                    ]],
                ]
            );
        }

        return null;
    }

    public function updateSubscriptionPaymentMethod($subscription_id, $payment_method_id)
    {
        $subscription = $this->getSubscription($subscription_id);

        if (isset($subscription->items->data[0]->id)) {
            return $this->client->subscriptions->update(
                $subscription_id, [
                    'default_payment_method' => $payment_method_id,
                ]
            );
        }

        return null;
    }

    /**
     * Preview Stripe Invoice
     *
     * @param $subscription_id
     * @param $new_price_id
     * @return \Stripe\Invoice|null
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function previewInvoice($subscription_id, $new_price_id)
    {
        $proration_date = time();

        $subscription = $this->getSubscription($subscription_id);

        if (isset($subscription->items->data[0]->id)) {
            $items = [
                [
                    'id' => $subscription->items->data[0]->id,
                    'price' => $new_price_id, # Switch to new price
                ],
            ];

            return $this->client->invoices->upcoming([
                'customer' => $subscription->customer,
                'subscription' => $subscription_id,
                'subscription_items' => $items,
                'subscription_proration_date' => $proration_date,
            ]);
        }

        return null;
    }
}