<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */
namespace app\modules\subscription\models\search;

use Yii;
use yii\base\Model;
use yii\data\ActiveDataProvider;
use app\modules\subscription\models\SubscriptionTransaction;

/**
 * SubscriptionTransactionSearch represents the model behind the search form of `app\modules\subscription\models\SubscriptionTransaction`.
 */
class SubscriptionTransactionSearch extends SubscriptionTransaction
{
    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['id', 'subscription_id'], 'integer'],
            [['gateway', 'gateway_id', 'gateway_status', 'gateway_transaction_id'], 'safe'],
            [['total'], 'number'],
            [['gateway_time'], 'safe'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function scenarios()
    {
        // bypass scenarios() implementation in the parent class
        return Model::scenarios();
    }

    /**
     * Creates data provider instance with search query applied
     *
     * @param array $params
     *
     * @return ActiveDataProvider
     */
    public function search($params)
    {
        $query = SubscriptionTransaction::find();

        // add conditions that should always apply here

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => Yii::$app->user->preferences->get('GridView.pagination.pageSize'),
            ],
            'sort' => [
                'defaultOrder' => [
                    'gateway_time' => SORT_DESC,
                ]
            ],
        ]);

        $this->load($params);

        if (!$this->validate()) {
            // uncomment the following line if you do not want to return any records when validation fails
            // $query->where('0=1');
            return $dataProvider;
        }

        // grid filtering conditions
        $query->andFilterWhere([
            'id' => $this->id,
            'subscription_id' => $this->subscription_id,
            'total' => $this->total,
        ]);

        if (!empty($this->gateway_time)) {
            list($start, $end) = explode(" - ", $this->gateway_time);
            $startAt = strtotime(trim($start));
            // Add +1 day to the endAt
            $endAt = strtotime(trim($end)) + (24 * 60 * 60);
            $query->andFilterWhere(['between', '{{%subscription_transaction}}.gateway_time', $startAt, $endAt]);
        }

        $query->andFilterWhere(['like', 'gateway', $this->gateway])
            ->andFilterWhere(['like', 'gateway_id', $this->gateway_id])
            ->andFilterWhere(['like', 'gateway_status', $this->gateway_status])
            ->andFilterWhere(['like', 'gateway_transaction_id', $this->gateway_transaction_id]);

        // Important! Don't remove this line
        $query->andFilterWhere(['{{%subscription_transaction}}.created_by' => Yii::$app->user->id]);

        return $dataProvider;
    }
}