<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\zoho_crm\services;

use app\helpers\ArrayHelper;
use Yii;
use yii\helpers\Url;

/**
 * Class ZohoCrmService
 * @package app\modules\addons\modules\zoho_crm\services
 */
class ZohoCrmService
{
    public $authUrl = 'https://accounts.zoho.com/oauth/v2/auth';
    public $tokenEndPoint = '/oauth/v2/token';
    public $authScope = 'ZohoCRM.settings.ALL,ZohoCRM.modules.all';
    public $redirectUri = null;
    public $accountsUrl = null;

    private $clientID;
    private $clientSecret;
    private $accessToken;
    private $apiDomain;

    /**
     * ZohoCrmService constructor.
     */
    public function __construct()
    {
        $this->clientID = Yii::$app->settings->get('addon_zoho_crm.clientId');
        $this->clientSecret = Yii::$app->settings->get('addon_zoho_crm.clientSecret');
        $this->redirectUri = Url::to(['/addons/zoho_crm/admin/authorized'], true);
    }

    /**
     * Redirect to Authentication URL
     * @throws \Exception
     */
    public function redirectToAuthUrl()
    {
        $auth_url = "{$this->authUrl}?scope={$this->authScope}&client_id={$this->clientID}&response_type=code&access_type=offline&redirect_uri={$this->redirectUri}";

        header('Location: ' . filter_var($auth_url, FILTER_SANITIZE_URL));
        exit;
    }

    /**
     * Set Accounts URL
     * Used to generate access and refresh tokens
     *
     * @param $accountsUrl
     */
    public function setAccountsUrl($accountsUrl)
    {
        $this->accountsUrl = $accountsUrl;
    }

    /**
     * Get Accounts URL
     *
     * @return null
     * @throws \Exception
     */
    public function getAccountsUrl()
    {
        if (empty($this->accountsUrl)) {
            throw new \Exception('Invalid Accounts URL.');
        }

        return $this->accountsUrl;
    }

    /**
     * Get Token URL
     *
     * @return string
     * @throws \Exception
     */
    public function getTokenUrl()
    {
        return $this->getAccountsUrl() . $this->tokenEndPoint;
    }

    /**
     * Fetch Access Token With Auth Code
     *
     * @param $code
     * @return array
     * @throws \Exception
     */
    public function fetchAccessTokenWithAuthCode($code)
    {
        $post = [
            'code'          => $code,
            'redirect_uri'  => $this->redirectUri,
            'client_id'     => $this->clientID,
            'client_secret' => $this->clientSecret,
            'grant_type'    => 'authorization_code',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->getTokenUrl());
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);

        $response = curl_exec($ch);

        return is_array($response) ? $response : json_decode($response, true);
    }

    /**
     * Fetch Access Token With Refresh Token
     *
     * @param $token
     * @return array
     * @throws \Exception
     */
    public function fetchAccessTokenWithRefreshToken($token)
    {
        if (empty($token)) {
            throw new \Exception('Invalid refresh token');
        }

        $post = [
            'refresh_token' => $token,
            'client_id'     => $this->clientID,
            'client_secret' => $this->clientSecret,
            'grant_type'    => 'refresh_token',
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->getTokenUrl());
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);

        $response = curl_exec($ch);

        return is_array($response) ? $response : json_decode($response, true);
    }

    /**
     * Set Access Token
     *
     * @param $accessToken
     */
    public function setAccessToken($accessToken)
    {
        $this->accessToken = $accessToken;
    }

    /**
     * Get Access Token
     *
     * @return string
     * @throws \Exception
     */
    public function getAccessToken()
    {
        if (empty($this->accessToken)) {
            throw new \Exception('Invalid Access Token');
        }

        return $this->accessToken;
    }

    /**
     * Set API Domain
     *
     * @param $apiDomain
     */
    public function setApiDomain($apiDomain)
    {
        $this->apiDomain = $apiDomain;
    }

    /**
     * Get API Domain
     *
     * @return mixed
     * @throws \Exception
     */
    public function getApiDomain()
    {
        if (empty($this->apiDomain)) {
            throw new \Exception('Invalid API Domain');
        }

        return $this->apiDomain;
    }

    /**
     * Get Modules
     *
     * https://developers.google.com/drive/api/v3/reference/files/list
     *
     * @return array
     * @throws \Exception
     */
    public function getModules()
    {
        $domain = $this->getApiDomain();
        $accessToken = $this->getAccessToken();
        $url = "{$domain}/crm/v2/settings/modules";

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL,$url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "GET");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Zoho-oauthtoken ' . $accessToken,
            'Content-Type: application/x-www-form-urlencoded',
        ]);

        $response = curl_exec($ch);
        $response = is_array($response) ? $response : json_decode($response, true);

        $modules = [];
        if (isset($response['modules']) && is_array($response['modules'])) {
            foreach ($response['modules'] as $module) {
                if (isset($module['module_name'], $module['api_name'], $module['api_supported']) && $module['api_supported'] === true) {
                    $modules[] = $module;
                }
            }

            $modules = ArrayHelper::map($modules, 'api_name', 'module_name');
        } else {
            throw new \Exception(json_encode($response));
        }

        return $modules;
    }

    /**
     * Get Fields
     *
     * @param $module
     * @return array|false
     * @throws \Exception
     */
    public function getFields($module)
    {
        $domain = $this->getApiDomain();
        $accessToken = $this->getAccessToken();
        $url = "{$domain}/crm/v2/settings/fields?module={$module}";

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL,$url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "GET");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Zoho-oauthtoken ' . $accessToken,
            'Content-Type: application/x-www-form-urlencoded',
        ]);

        $response = curl_exec($ch);
        $response = is_array($response) ? $response : json_decode($response, true);

        $fields = [];
        if (isset($response['fields']) && is_array($response['fields'])) {
            foreach ($response['fields'] as $field) {
                if (isset($field['display_label'], $field['api_name'], $field['system_mandatory'])) {

                    $flag = '';

                    if (isset($field['system_mandatory']) && $field['system_mandatory'] === true) {
                        $flag .= ' - ' . Yii::t('app', 'Required');
                    }

                    if (isset($field['unique']) && $field['unique'] === true) {
                        $flag .= ' - ' . Yii::t('app', 'Unique');
                    }

                    if (isset($field['read_only']) && $field['read_only'] === true) {
                        $flag .= ' - ' . Yii::t('app', 'Read Only');
                    }

                    if (isset($field['length'])) {
                        $flag .= ' - ' . Yii::t('app', 'Max Length') . ' ' . $field['length'];
                    }

                    $fields[] = [
                        'id' => $field['api_name'],
                        'name' => $field['display_label'] . $flag,
                    ];
                }
            }
        } else {
            throw new \Exception(json_encode($response));
        }

        return $fields;
    }

    /**
     * Insert Record
     *
     * @param $module
     * @param $data
     * @param $trigger
     * @return array|bool|mixed|string
     * @throws \Exception
     */
    public function insert($module, $data, $trigger)
    {
        $post = [
            'data' => [
                $data
            ],
            'trigger' => $trigger,
        ];

        $domain = $this->getApiDomain();
        $accessToken = $this->getAccessToken();
        $url = "{$domain}/crm/v2/{$module}";

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($post));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Zoho-oauthtoken ' . $accessToken,
            'Content-Type: application/x-www-form-urlencoded',
        ]);

        $response = curl_exec($ch);

        return is_array($response) ? $response : json_decode($response, true);
    }

    /**
     * Upsert Record
     *
     * @param $module
     * @param $data
     * @param $trigger
     * @param $duplicate_check_fields
     * @return array|bool|mixed|string
     * @throws \Exception
     */
    public function upsert($module, $data = [], $trigger = [], $duplicate_check_fields = [])
    {
        $post = [
            'data' => [
                $data
            ],
            "duplicate_check_fields" => $duplicate_check_fields,
            'trigger' => $trigger,
        ];

        $domain = $this->getApiDomain();
        $accessToken = $this->getAccessToken();
        $url = "{$domain}/crm/v2/{$module}/upsert";

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($post));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Zoho-oauthtoken ' . $accessToken,
            'Content-Type: application/x-www-form-urlencoded',
        ]);

        $response = curl_exec($ch);

        return is_array($response) ? $response : json_decode($response, true);
    }

}