<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\zoho_crm\models;

use app\models\User;
use Yii;
use app\models\Form;
use app\components\behaviors\RelationTrait;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;
use app\components\behaviors\DateTrait;
use yii\helpers\Json;

/**
 * This is the model class for table "{{%addon_zoho_crm}}".
 *
 * @property int $id
 * @property int $form_id
 * @property int $status
 * @property string $name
 * @property int $event
 * @property string|null $conditions
 * @property string $oauth
 * @property string $accounts_url
 * @property string|null $module
 * @property string|null $trigger
 * @property int $approved
 * @property int $action
 * @property string|null $duplicate_check_fields
 * @property int|null $created_by
 * @property int|null $updated_by
 * @property int|null $created_at
 * @property int|null $updated_at
 *
 * @property Form $form
 * @property User $author
 * @property User $lastEditor
 * @property ZohoCrmItem[] $items
 */
class ZohoCrm extends \yii\db\ActiveRecord
{
    use RelationTrait, DateTrait;

    const OFF = 0;
    const ON = 1;

    const ACTION_INSERT = 1;
    const ACTION_UPDATE = 3;
    const ACTION_UPSERT = 5;

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%addon_zoho_crm}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['form_id', 'name', 'module'], 'required'],
            [['form_id', 'status', 'event', 'approved', 'action', 'created_by', 'updated_by', 'created_at', 'updated_at'], 'integer'],
            [['conditions', 'oauth', 'accounts_url'], 'string'],
            [['name'], 'string', 'max' => 255],
            [['trigger', 'duplicate_check_fields'], 'safe'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'form_id' => Yii::t('app', 'Form ID'),
            'status' => Yii::t('app', 'Status'),
            'name' => Yii::t('app', 'Name'),
            'event' => Yii::t('app', 'Event'),
            'conditions' => Yii::t('app', 'Conditions'),
            'oauth' => Yii::t('app', 'Oauth'),
            'module' => Yii::t('app', 'Module'),
            'accounts_url' => Yii::t('app', 'Accounts URL'),
            'trigger' => Yii::t('app', 'Trigger'),
            'approved' => Yii::t('app', 'Approved'),
            'action' => Yii::t('app', 'Action'),
            'duplicate_check_fields' => Yii::t('app', 'Duplicate Check Fields'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }


    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForm()
    {
        return $this->hasOne(Form::class, ['id' => 'form_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getItems()
    {
        return $this->hasMany(ZohoCrmItem::class, ['form_id' => 'form_id']);
    }

    public function setItems($value)
    {
        $this->loadRelated('items', $value);
    }

    /**
     * {@inheritdoc}
     */
    public function beforeValidate()
    {
        if (!empty($this->trigger) && is_array($this->trigger)) {
            $this->trigger = Json::encode($this->trigger, true);
        }
        if (!empty($this->duplicate_check_fields) && is_array($this->duplicate_check_fields)) {
            $this->duplicate_check_fields = Json::encode($this->duplicate_check_fields, true);
        }
        return parent::beforeValidate();
    }

    /**
     * @inheritdoc
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            ZohoCrmItem::deleteAll(["zoho_crm_id" => $this->id]);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Get Triggers
     *
     * @return string[]
     */
    public static function triggerDropdown()
    {
        return [
            'workflow',
            'approval',
            'blueprint',
        ];
    }

    /**
     * Get Actions
     *
     * @return array
     */
    public static function actionDropdown()
    {
        return [
            self::ACTION_UPSERT => Yii::t('app', 'Insert or Update Records (Upsert)'),
            self::ACTION_INSERT => Yii::t('app', 'Insert Records'),
        ];
    }
}