<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\zoho_crm\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\modules\addons\models\Addon;
use app\modules\addons\modules\zoho_crm\models\ZohoCrm;
use app\modules\addons\modules\zoho_crm\models\ZohoCrmItem;
use app\modules\addons\modules\zoho_crm\models\ZohoCrmSearch;
use app\modules\addons\modules\zoho_crm\services\ZohoCrmService;
use kartik\depdrop\DepDropAction;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\Html;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\Controller;
use yii\web\NotFoundHttpException;

/**
 * AdminController implements the CRUD actions for Ga model.
 */
class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'modules', 'module-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['settings'], 'allow' => true, 'roles' => ['configureAddons'], 'roleParams' => function() {
                        return ['model' => Addon::findOne(['id' => 'zoho_crm'])];
                    }],
                    ['actions' => ['create', 'authorize', 'authorized'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } elseif ($postData = Yii::$app->request->post('ZohoCrm')) {
                            if (!empty($postData['form_id']) ) {
                                return ['model' => Form::findOne(['id' => $postData['form_id']])];
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = ZohoCrm::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\zoho_crm\models\ZohoCrm',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                $fields = SubmissionHelper::getFieldsForFieldMapping($fields);
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'modules' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($module, $params) {
                    $output = array();
                    $token = null;
                    $formID = null;
                    if (isset($_POST['depdrop_all_params']['zohocrm-form_id'])) {
                        $formID = $_POST['depdrop_all_params']['zohocrm-form_id'];
                    }
                    $model = ZohoCrm::findOne(['form_id' => $formID]);

                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                            $token = !is_string($model->oauth) ? $model->oauth : Json::decode($model->oauth);
                        }
                    } elseif (Yii::$app->session->has('zoho_crm_access_token')) {
                        $token = Yii::$app->session->get('zoho_crm_access_token');
                    }

                    if ($token && $module) {
                        $service = new ZohoCrmService();
                        $service->setAccessToken($token);
                        $output = $service->getSheetList($module);
                    }

                    return $output;
                },
                'selectedCallback' => function ($id, $params) {
                    if (isset($_POST['depdrop_params'])) {
                        $ids = $_POST['depdrop_params'];
                        return empty($ids[0]) ? null : $ids[0];
                    }
                }
            ],
            'module-fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function () {
                    $output = array();
                    if (isset($_POST['depdrop_parents'])) {
                        $ids = $_POST['depdrop_parents'];
                        $module = empty($ids[0]) ? null : $ids[0];
                        $token = null;
                        $accountsUrl = null;
                        $formID = null;
                        if (isset($_POST['depdrop_all_params']['zohocrm-form_id'])) {
                            $formID = $_POST['depdrop_all_params']['zohocrm-form_id'];
                        }
                        $model = ZohoCrm::findOne(['form_id' => $formID]);
                        if (!empty($model->form)) {
                            if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                                $token = !is_string($model->oauth) ? $model->oauth : Json::decode($model->oauth);
                                $accountsUrl = $model->accounts_url;
                            }
                        } elseif (Yii::$app->session->has('zoho_crm_access_token') && Yii::$app->session->has('zoho_crm_accounts_url')) {
                            $token = Yii::$app->session->get('zoho_crm_access_token');
                            $accountsUrl = Yii::$app->session->get('zoho_crm_accounts_url');
                        }

                        if ($token && $module) {
                            $service = new ZohoCrmService();
                            $service->setAccountsUrl($accountsUrl);
                            $response = $service->fetchAccessTokenWithRefreshToken($token['refresh_token']);
                            if (isset($response['access_token'], $response['api_domain'])) {
                                $service->setAccessToken($response['access_token']);
                                $service->setApiDomain($response['api_domain']);
                                $output = $service->getFields($module);
                            }
                        }
                    }

                    return $output;
                },
                'selectedCallback' => function ($id, $params) {
                    if (isset($_POST['depdrop_params'])) {
                        $ids = $_POST['depdrop_params'];
                        return empty($ids[0]) ? null : $ids[0];
                    }
                }
            ],
        ];
    }

    /**
     * Lists all Zoho CRM models.
     * @return string
     */
    public function actionIndex()
    {
        $searchModel = new ZohoCrmSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single ZohoCrm model.
     *
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new ZohoCrm model.
     * If creation is successful, the browser will be redirected to the 'index' page.
     * @return string
     */
    public function actionCreate()
    {

        $model = new ZohoCrm();
        $itemModel = new ZohoCrmItem();
        $token = Yii::$app->session->get('zoho_crm_access_token');
        $accountsUrl = Yii::$app->session->get('zoho_crm_accounts_url');
        $token = is_string($token) ? json_decode($token, true) : $token;
        $modules = [];

        try {
            if ($token) {

                // Connect to Zoho CRM
                $service = new ZohoCrmService();
                $service->setAccountsUrl($accountsUrl);
                $response = $service->fetchAccessTokenWithRefreshToken($token['refresh_token']);
                $service->setAccessToken($response['access_token']);
                $service->setApiDomain($response['api_domain']);
                $modules = $service->getModules();

                if ($model->load(Yii::$app->request->post())) {
                    $items = Yii::$app->request->post('ZohoCrmItem',[]);
                    if (!empty($items)) {
                        $model->oauth = !is_string($token) ? Json::encode($token) : $token;
                        $model->accounts_url = $accountsUrl;
                        if ($model->validate()) {
                            $model->items = $items;
                            $model->save(false);
                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );
                            return $this->redirect(['index']);
                        } else {
                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                            );
                        }
                    }
                }
            }

        } catch (\Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'itemModel' => $itemModel,
            'forms' => $forms,
            'modules' => $modules,
            'token' => $token,
        ]);
    }

    /**
     * Authorize Zoho Account
     *
     * @param null $id
     * @throws \Exception
     */
    public function actionAuthorize($id = null)
    {
        $url = Url::to(['/addons/zoho_crm/admin/create'], true);

        if ($id) {
            $url = Url::to(['/addons/zoho_crm/admin/update', 'id' => $id], true);
        }

        Yii::$app->session->set('redirect_url', $url);

        $service = new ZohoCrmService();
        $service->redirectToAuthUrl();
    }

    /**
     * Get Access Token form Authorized Zoho Account
     *
     * @return \yii\web\Response
     * @throws \Exception
     */
    public function actionAuthorized()
    {
        $request = Yii::$app->request;
        $session = Yii::$app->session;

        $error = $request->get('error'); // E.g. 'access_denied'
        $code = $request->get('code'); // To generate access and refresh tokens.
        $location = $request->get('location'); // Domain of the user from which you have to make API calls
        $accountsServer = $request->get('accounts-server'); // Accounts URL which you have to use to generate access and refresh tokens.

        $url = $session->get('redirect_url');

        if (empty($error) && !empty($code)) {
            $service = new ZohoCrmService();
            $service->setAccountsUrl($accountsServer);
            $token = $service->fetchAccessTokenWithAuthCode($code);
            // Save token and accounts url in user session
            $session->set('zoho_crm_access_token', $token);
            $session->set('zoho_crm_accounts_url', $accountsServer);

            if (empty($url)) {
                Yii::$app->getSession()->setFlash(
                    'success',
                    Yii::t('app', 'Your access token has been successfully configured.')
                );
                $url = ['index'];
            }

            return $this->redirect($url);
        }
    }

    /**
     * Updates an existing ZohoCrm model.
     * If update is successful, the browser will be redirected to the 'index' page.
     *
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $session = Yii::$app->session;
        $modules = null;
        $token = !empty($model->oauth) && is_string($model->oauth) ? Json::decode($model->oauth) : $model->oauth;

        if ($session->has('zoho_crm_access_token')) {
            $token = $session->get('zoho_crm_access_token');
            $model->oauth = Json::encode($token);
            $model->save();
        }

        if ($session->has('zoho_crm_accounts_url')) {
            $model->accounts_url = $session->get('zoho_crm_accounts_url');
            $model->save();
        }

        try {
            // Connect to Zoho CRM
            $service = new ZohoCrmService();
            $service->setAccountsUrl($model->accounts_url);
            $response = $service->fetchAccessTokenWithRefreshToken($token['refresh_token']);
            $service->setAccessToken($response['access_token']);
            $service->setApiDomain($response['api_domain']);
            $modules = $service->getModules();

            if ($model->load(Yii::$app->request->post())) {
                $items = Yii::$app->request->post('ZohoCrmItem',[]);
                if (!empty($items)) {
                    if ($model->validate()) {
                        $model->items = $items;
                        $model->save(false);
                        Yii::$app->getSession()->setFlash(
                            'success',
                            Yii::t('app', 'The form has been successfully configured.')
                        );
                        return $this->redirect(['index']);
                    } else {
                        // Show error message
                        Yii::$app->getSession()->setFlash(
                            'danger',
                            Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                        );
                    }
                }
            }

        } catch (\Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            $refreshLink = ' - ' . Yii::t('app', 'Authenticate your Zoho CRM account to update your integration.');
			if (!empty($model->oauth)) {
                $refreshLink = ' - ' . Html::a(Yii::t('app', 'Authenticate your Zoho CRM account to update your integration.'), Url::current(), ['class' => 'text-danger']);
            }
            Yii::$app->getSession()->setFlash('danger', $e->getMessage() . ' ' . $refreshLink);
            // Reset access token
            $model->oauth = null;
            $model->accounts_url = null;
            $model->save();
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'itemModel' => new ZohoCrmItem,
            'forms' => $forms,
            'modules' => $modules,
            'token' => $token,
        ]);
    }

    /**
     * Global settings for Zoho CRM integration
     *
     * @return string
     */
    public function actionSettings()
    {
        $request = Yii::$app->request;
        $settings = Yii::$app->settings;

        if ($request->post()) {
            if ($request->post('action') === 'global-settings') {
                try {

                    $clientID = strip_tags($request->post('addon_zoho_crm_clientId', $settings->get('addon_zoho_crm.clientId')));
                    $clientSecret = strip_tags($request->post('addon_zoho_crm_clientSecret', $settings->get('addon_zoho_crm_clientSecret')));
                    $settings->set('addon_zoho_crm.clientId', $clientID);
                    $settings->set('addon_zoho_crm.clientSecret', $clientSecret);
                    // Show success alert
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'Zoho CRM Integration Settings have been successfully updated.')
                    );
                } catch (\Exception $e) {
                    // Clear settings
                    $settings->set('addon_zoho_crm.clientId', '');
                    $settings->set('addon_zoho_crm.clientSecret', '');
                    // Log
                    Yii::error($e);
                    // Show error message
                    Yii::$app->getSession()->setFlash('danger', $e->getMessage());
                }
            }
        }

        return $this->render('settings');
    }

    /**
     * Enable / Disable multiple Zoho Analytics ZohoCrm
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function actionUpdateStatus($status)
    {
        $models = ZohoCrm::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        } else {
            foreach ($models as $model) {
                $model->status = $status;
                $model->update();
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The selected items have been successfully updated.')
            );
            return $this->redirect(['index']);
        }
    }

    /**
     * Deletes an existing ZohoCrm model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     *
     * @param $id
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        Yii::$app->getSession()->setFlash('success', Yii::t('app', 'The configuration has been successfully deleted.'));

        return $this->redirect(['index']);
    }

    /**
     * Finds the ZohoCrm model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return ZohoCrm the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = ZohoCrm::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
