<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\zoho_crm;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\zoho_crm\models\ZohoCrm;
use app\modules\addons\modules\zoho_crm\models\ZohoCrmItem;
use app\modules\addons\modules\zoho_crm\services\ZohoCrmService;
use Exception;
use Yii;
use yii\helpers\Json;

/**
 * Class Module
 * @package app\modules\addons\modules\zoho_crm
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "zoho_crm";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'ZohoCrm' => ZohoCrm::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = ZohoCrm::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new ZohoCrm();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new ZohoCrmItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->zoho_crm_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * On Form Before Delete
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = ZohoCrm::findAll(['form_id' => $event->sender->id]);
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getUrl();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getUrl();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {
        $result = false;

        $models = ZohoCrm::findAll(['form_id' => $formModel->id, 'status' => 1]);
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                try {


                    $data = [];
                    foreach ($model->items as $item) {

                        $value = isset($submissionData[$item->field_id]) ? $submissionData[$item->field_id] : '';
                        $data[$item->module_field] = $value;

                    }

                    $trigger = !empty($model->trigger) ? Json::decode($model->trigger, true) : [];
                    $duplicateCheckFields = !empty($model->duplicate_check_fields) ? Json::decode($model->duplicate_check_fields, true) : [];

                    // Connect to Zoho CRM
                    $token = !empty($model->oauth) && is_string($model->oauth) ? Json::decode($model->oauth) : $model->oauth;

                    $service = new ZohoCrmService();
                    $service->setAccountsUrl($model->accounts_url);
                    $response = $service->fetchAccessTokenWithRefreshToken($token['refresh_token']);
                    $service->setAccessToken($response['access_token']);
                    $service->setApiDomain($response['api_domain']);

                    if ($model->action === ZohoCrm::ACTION_UPSERT) {

                         $service->upsert($model->module, $data, $trigger, $duplicateCheckFields);

                    } elseif ($model->action === ZohoCrm::ACTION_INSERT) {

                        $service->insert($model->module, $data, $trigger);

                    }

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;
    }
}