<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */
namespace app\modules\subscription\models;

use app\components\behaviors\DateTrait;
use app\components\behaviors\RelationTrait;
use app\models\User;
use Yii;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;
use yii\helpers\ArrayHelper;

/**
 * This is the model class for table "{{%subscription_product}}".
 *
 * @property int $id
 * @property string $name
 * @property string|null $description
 * @property int $status
 * @property string|null $details
 * @property int $recommended
 * @property int $form_limit
 * @property int $submission_limit
 * @property int $file_upload_limit
 * @property int $file_storage_limit
 * @property int $api_request_limit
 * @property int $branding
 * @property int $created_by
 * @property int $updated_by
 * @property int|null $created_at
 * @property int|null $updated_at
 *
 * @property SubscriptionPrice[] $items
 * @property SubscriptionProductUserRole[] $subscriptionProductRoles
 */
class SubscriptionProduct extends \yii\db\ActiveRecord
{
    use RelationTrait, DateTrait;

    const ON = 1;
    const OFF = 0;

    // Used in Global Settings
    const MODE_TEST = 'sandbox';
    const MODE_LIVE = 'production';

    public $roles;

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%subscription_product}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['name'], 'required'],
            [['description', 'details'], 'string'],
            [['status',
                'recommended', 'form_limit', 'submission_limit', 'file_upload_limit',
                'file_storage_limit', 'api_request_limit', 'branding',
                'created_by', 'updated_by', 'created_at', 'updated_at'], 'integer'],
            [['name'], 'string', 'max' => 255],
            [['form_limit', 'submission_limit'], 'number', 'min' => 1, 'max' => 99999999999],
            [['items'], 'safe'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'name' => Yii::t('app', 'Name'),
            'description' => Yii::t('app', 'Description'),
            'status' => Yii::t('app', 'Status'),
            'details' => Yii::t('app', 'Details'),
            'recommended' => Yii::t('app', 'Recommended'),
            'form_limit' => Yii::t('app', 'Form Limit'),
            'submission_limit' => Yii::t('app', 'Submission Limit'),
            'file_upload_limit' => Yii::t('app', 'File Upload Limit'),
            'file_storage_limit' => Yii::t('app', 'File Storage Limit'),
            'api_request_limit' => Yii::t('app', 'Api Request Limit'),
            'branding' => Yii::t('app', 'Branding'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSubscriptionProductRoles()
    {
        return $this->hasMany(SubscriptionProductUserRole::class, ['product_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getItems()
    {
        return $this->hasMany(SubscriptionPrice::class, ['product_id' => 'id']);
    }

    public function setItems($value)
    {
        $this->loadRelated('items', $value);
    }

    /**
     * @inheritdoc
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            SubscriptionPrice::deleteAll(["product_id" => $this->id]);
            SubscriptionProductUserRole::deleteAll(["product_id" => $this->id]);
            return true;
        }

        return false;
    }

    /**
     * Get Mode list
     *
     * @return array
     */
    public static function availableModes()
    {
        return [
            self::MODE_TEST => Yii::t('app', 'Test / Sandbox'),
            self::MODE_LIVE => Yii::t('app', 'Live'),
        ];
    }

    /**
     * Get Role IDs
     *
     * @return array User Role IDs
     */
    public function getRoles()
    {
        $productRoles = $this->getSubscriptionProductRoles()->asArray()->all();
        return ArrayHelper::getColumn($productRoles, 'role_id');
    }
}
