<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\stripe\services;

use Yii;

class StripeService
{
    public function __construct($apiKey)
    {
        // Set your secret key: remember to change this to your live secret key in production
        // See your keys here: https://dashboard.stripe.com/account/apikeys
        \Stripe\Stripe::setApiKey($apiKey);
    }

    /**
     * API call to charge the card
     *
     * Params Eg.
     * "amount" => 1000,
     * "currency" => "usd",
     * "description" => "Example charge",
     * "source" => $token,
     *
     * @param array $params
     * @return \Stripe\Charge
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function charge(array $params)
    {
//        $paymentIntent = $stripe->paymentIntents->create([
//            'amount' => 1099,
//            'currency' => 'eur',
//            'customer' => $customer->id,
//            // In the latest version of the API, specifying the `automatic_payment_methods` parameter
//            // is optional because Stripe enables its functionality by default.
//            'automatic_payment_methods' => [
//                'enabled' => 'true',
//            ],
//            'application_fee_amount' => 123,
//            'transfer_data' => [
//                'destination' => '{{CONNECTED_ACCOUNT_ID}}',
//            ],
//        ]);
//
//
//        return \Stripe\PaymentIntent::create($params);

        // Charge the user's card:
        return \Stripe\Charge::create($params);
    }

    /**
     * API call to subscribe using the card
     *
     * Params Eg.
     * "amount" => 1000,
     * "currency" => "usd",
     * "description" => "Example charge",
     * "source" => $token,
     *
     * @param array $params
     * @return \Stripe\Subscription
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function subscribe(array $params)
    {
        // Charge the user's card:
        return \Stripe\Subscription::create($params);
    }

    /**
     * Create customer
     *
     * @param array $params
     * @return \Stripe\Customer
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function customer(array $params)
    {
        return \Stripe\Customer::create($params);
    }

    /**
     * API call to find a customer
     *
     * Params Eg.
     * "email" => "john.doe@example.com",
     *
     * @param array $params
     * @return string
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function findCustomerIdBy(array $params)
    {
        $params = array_merge(['limit' => 1], $params);
        $customers = \Stripe\Customer::all($params);
        if (!isset($customers['data'], $customers['data'][0]) && !empty($customers['data'][0]->id)) {
            return $customers['data'][0]->id;
        }

        return '';
    }

    /**
     * Create Checkout Session
     *
     * @param $params
     * @return \Stripe\Checkout\Session
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function session($params)
    {
        return \Stripe\Checkout\Session::create($params);
    }

    /**
     * Retrieve Checkout Session By ID
     *
     * @param $id
     * @return \Stripe\Checkout\Session
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveSessionByById($id)
    {
        return \Stripe\Checkout\Session::retrieve($id);
    }

    /**
     * Retrieve Stripe Price By ID
     *
     * @param $price_id
     * @return \Stripe\Price
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrievePrice($price_id) {
        return \Stripe\Price::retrieve($price_id);
    }

    /**
     * Retrieve Stripe Price By ID
     *
     * @param $promo_code
     * @return \Stripe\Collection
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveDiscount($promo_code) {

        return \Stripe\PromotionCode::all(['code' => $promo_code]);
    }


    /**
     * Retrieve Stripe Price By ID
     *
     * @param $promo_code_id
     * @return \Stripe\PromotionCode
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveDiscountByID($promo_code_id) {
        return \Stripe\PromotionCode::retrieve($promo_code_id);
    }

    /**
     * Retrieve Payment Intent By ID
     *
     * @param $id
     * @return \Stripe\PaymentIntent
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrievePaymentIntent($id)
    {
        return \Stripe\PaymentIntent::retrieve($id);
    }

    /**
     * Retrieve Subscription By ID
     *
     * @param $id
     * @return \Stripe\Subscription
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveSubscription($id)
    {
        return \Stripe\Subscription::retrieve($id);
    }

    /**
     * Create a Webhook Endpoint
     *
     * @param $url
     * @param $events
     * @return \Stripe\WebhookEndpoint
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function createWebhookEndpoint($url, $events) {
        return \Stripe\WebhookEndpoint::create([
            'url' => $url,
            'enabled_events' => $events,
        ]);
    }

    /**
     * Retrieve a Webhook Endpoint
     *
     * @param $url
     * @param $events
     * @return \Stripe\WebhookEndpoint
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function retrieveWebhookEndpoint($url, $events) {
        return \Stripe\WebhookEndpoint::create([
            'url' => $url,
            'enabled_events' => $events,
        ]);
    }

    /**
     * Verify with Stripe that the event is genuine.
     *
     * @param string $id
     *
     * @return \Stripe\Event
     */
    public function eventExistsOnStripe($id)
    {
        try {

            return \Stripe\Event::retrieve($id);

        } catch (\Exception $e) {

            Yii::error($e);

        }
    }
}