<?php
namespace app\modules\addons\modules\smtp\models;

use app\components\behaviors\DateTrait;
use app\helpers\ArrayHelper;
use app\models\Form;
use app\models\User;
use app\modules\addons\HasFormsInterface;
use Yii;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;

/**
 * This is the model class for table "{{%addon_smtp}}".
 *
 * @property integer $id
 * @property integer $status
 * @property string $name
 * @property string $conditions
 * @property integer $event
 * @property string $host
 * @property string $port
 * @property string $encryption
 * @property string $auth
 * @property string $username
 * @property string $password
 * @property string $email_address
 * @property string $from_name
 * @property integer $created_by
 * @property integer $updated_by
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property array $selected_forms
 * @property Form[] $forms
 * @property SmtpForm[] $smtpForms
 * @property User $author
 * @property User $lastEditor
 */
class Smtp extends \yii\db\ActiveRecord implements HasFormsInterface
{
    use DateTrait;

    const ON = 1;
    const OFF = 0;

    const ENCRYPTION_NONE = 'none';
    const ENCRYPTION_SSL = 'ssl';
    const ENCRYPTION_TLS = 'tls';

    public $selected_forms;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%addon_smtp}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['name', 'selected_forms', 'host', 'port'], 'required'],
            [['status', 'event', 'created_by', 'updated_by', 'created_at', 'updated_at'], 'integer'],
            [['name', 'conditions', 'host', 'port', 'encryption', 'auth', 'username', 'password', 'email_address', 'from_name'], 'string'],
            ['selected_forms', 'validateForms'],
        ];
    }

    /**
     * A form can only be linked to one SMTP settings
     */
    public function validateForms()
    {
        /** @var SmtpForm[] $smtpForms */
        $smtpForms = SmtpForm::find()
            ->where(['in', 'form_id', $this->selected_forms])
            ->andWhere(['!=', 'smtp_id', $this->id])
            ->all();

        foreach ($smtpForms as $smtpForm) {
            $this->addError('forms', Yii::t('app', 'The "{form}" form has been selected in another SMTP settings.', [
                'form' => $smtpForm->form->name,
            ]));
        }
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'status' => Yii::t('app', 'Status'),
            'name' => Yii::t('app', 'Name'),
            'conditions' => Yii::t('app', 'Conditions'),
            'event' => Yii::t('app', 'Event'),
            'host' => Yii::t('app', 'Host'),
            'port' => Yii::t('app', 'Port'),
            'encryption' => Yii::t('app', 'Encryption'),
            'auth' => Yii::t('app', 'Authentication Method'),
            'username' => Yii::t('app', 'Username'),
            'password' => Yii::t('app', 'Password'),
            'email_address' => Yii::t('app', 'Email Address'),
            'from_name' => Yii::t('app', 'From Name'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSmtpForms()
    {
        return $this->hasMany(SmtpForm::class, ['smtp_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForms()
    {
        return $this->hasMany(Form::class, ['id' => 'form_id'])
            ->via('smtpForms');
    }

    /**
     * Find Models by Form ID
     *
     * @param integer $id Form ID
     * @return Smtp[]
     */
    public static function findAllByFormID($id)
    {
        $smtpIds = SmtpForm::find()
            ->select(['smtp_id'])
            ->where(['form_id' => $id])
            ->asArray()
            ->all();
        $smtpIds = ArrayHelper::getColumn($smtpIds, 'smtp_id');
        return self::findAll(['id' => $smtpIds]);
    }

    /**
     * @inheritdoc
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            SmtpForm::deleteAll(["smtp_id" => $this->id]);
            return true;
        } else {
            return false;
        }
    }

}