<?php

namespace app\modules\addons\modules\smtp\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\models\Form;
use app\modules\addons\modules\smtp\models\Smtp;
use app\modules\addons\modules\smtp\models\SmtpForm;
use app\modules\addons\modules\smtp\models\SmtpSearch;
use Exception;
use Swift_Mailer;
use Swift_SmtpTransport;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;

/**
 * AdminController implements the CRUD actions for Smtp model.
 */
class AdminController extends Controller
{
    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('Smtp')) {
                            if (isset($postData['selected_forms'])) {
                                if (is_array($postData['selected_forms'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['selected_forms']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        $smtpForms = $model->getSmtpForms()->asArray()->all();
                        $ids = ArrayHelper::getColumn($smtpForms, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = SmtpForm::find()
                            ->select(['form_id'])
                            ->where(['in', 'smtp_id', Yii::$app->request->post('ids')])
                            ->asArray()
                            ->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\smtp\models\smtp',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
        ];
    }

    /**
     * Lists all smtp models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new SmtpSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single smtp model.
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new smtp model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Smtp;
        $request = Yii::$app->request;

        if ($model->load($request->post())) {

            try {
                // Test SMTP connection
                $transport = new Swift_SmtpTransport($model->host, $model->port);
                if ($model->encryption !== Smtp::ENCRYPTION_NONE) {
                    $transport = new Swift_SmtpTransport($model->host, $model->port, $model->encryption);
                }
                $transport->setUsername($model->username);
                $transport->setPassword($model->password);
                $mailer = new Swift_Mailer($transport);
                $mailer->getTransport()->start();

                // Save SMTP Settings
                if ($model->save()) {
                    foreach ($model->selected_forms as $formID) {
                        $smtpForm = new SmtpForm;
                        $smtpForm->smtp_id = $model->id;
                        $smtpForm->form_id = $formID;
                        $smtpForm->save();
                    }
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'SMTP settings have been successfully saved.')
                    );
                }

                return $this->redirect(['index']);

            } catch (Exception $e) {
                // Log error
                Yii::error($e);
                // Show error alert
                Yii::$app->getSession()->setFlash(
                    'danger',
                    $e->getMessage()
                );
            }
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'forms' => $forms,
        ]);
    }

    /**
     * Updates an existing smtp model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $request = Yii::$app->request;

        if ($model->load($request->post())) {

            try {
                // Test SMTP connection
                $transport = new Swift_SmtpTransport($model->host, $model->port);
                if ($model->encryption !== Smtp::ENCRYPTION_NONE) {
                    $transport = new Swift_SmtpTransport($model->host, $model->port, $model->encryption);
                }
                $transport->setUsername($model->username);
                $transport->setPassword($model->password);
                $mailer = new Swift_Mailer($transport);
                $mailer->getTransport()->start();

                // Save SMTP Settings
                if ($model->save() && !empty($model->selected_forms)) {

                    SmtpForm::deleteAll(['smtp_id' => $model->id]);

                    foreach ($model->selected_forms as $formID) {
                        $smtpForm = new SmtpForm;
                        $smtpForm->smtp_id = $model->id;
                        $smtpForm->form_id = $formID;
                        $smtpForm->save();
                    }

                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'SMTP settings have been successfully updated.')
                    );

                    return $this->redirect(['index']);
                }

            } catch (Exception $e) {
                // Log error
                Yii::error($e);
                // Show error alert
                Yii::$app->getSession()->setFlash(
                    'danger',
                    $e->getMessage()
                );
            }
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        $model->selected_forms = $model->getSmtpForms()->asArray()->all();
        $model->selected_forms =ArrayHelper::getColumn($model->selected_forms, 'form_id');

        return $this->render('update', [
            'model' => $model,
            'forms' => $forms,
        ]);
    }

    /**
     * Enable / Disable multiple Smtp models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = Smtp::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = (int) $status;
            $model->save(false);
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing Smtp model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the smtp model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return smtp the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Smtp::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
