<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\smtp;

use app\components\queue\MailQueue;
use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\controllers\SubmissionsController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\smtp\models\Smtp;
use app\modules\addons\modules\smtp\models\SmtpForm;
use Exception;
use Yii;

/**
 * Class Module
 * @package app\modules\addons\modules\smtp
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "smtp";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'Smtp' => Smtp::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.received' => function ($event) {
                $this->onSubmissionReceived($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ],
            ],
            SubmissionsController::class => [
                'beforeAction' => [
                    [Module::class, 'onSubmissionEmail']
                ]
            ]
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = SmtpForm::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new SmtpForm();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = SmtpForm::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * When emails are sent with the Submission Manager
     *
     * @param $event
     */
    public static function onSubmissionEmail($event)
    {
        if (isset($event->action->controller->request)) {

            $formModel = Form::findOne(['id' => $event->action->controller->request->get('id')]);

            if ($formModel) {
                self::setSmtpServer($formModel);
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been received
     *
     * @param $event
     */
    public function onSubmissionReceived($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;

        if ($formModel) {
            self::setSmtpServer($formModel);
        }
    }

    /**
     * Set SMTP Server
     *
     * @param $formModel
     * @param $submissionModel
     * @return void
     */
    public static function setSmtpServer($formModel)
    {

        $models = SmtpForm::findAll(['form_id' => $formModel->id]);

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            try {

                // Only if the SMTP settings status is enabled
                if ($model->smtp->status !== Smtp::ON) {
                    continue;
                }

                Yii::$app->set('mailer', [
                    'class' => MailQueue::class,
                    'mailsPerRound' => 10,
                    'maxAttempts' => 3,
                    'transport' => [
                        'class' => 'Swift_SmtpTransport',
                        'host' => $model->smtp->host,
                        'username' => $model->smtp->username,
                        'password' => $model->smtp->password,
                        'port' => $model->smtp->port,
                        'encryption' => $model->smtp->encryption === "none" ? null : $model->smtp->encryption,
                    ],
                    'messageConfig' => [
                        'charset' => 'UTF-8',
                    ]
                ]);

                if (is_string($model->smtp->email_address)
                    && (!filter_var($model->smtp->email_address, FILTER_VALIDATE_EMAIL) === false)) {
                    // Overwrite always
                    // Yii::$app->mailer->messageConfig = [
                    //     'from' => $model->smtp->username,
                    // ];
                    // Overwrite dynamically
                    Yii::$app->params['App.Modules.SMTP.username'] = $model->smtp->email_address;
                }
                if (!empty(trim($model->smtp->from_name))) {
                    // Overwrite dynamically
                    Yii::$app->params['App.Modules.SMTP.fromName'] = $model->smtp->from_name;
                }

            } catch (Exception $e) {

                // Log exception
                Yii::error($e);

            }
        }
    }
}