<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\salesforce\services;

use Yii;
use yii\httpclient\Client;

/**
 * Class SalesforceService
 * @package app\modules\addons\modules\salesforce\services
 */
class SalesforceService
{
    protected $client;
    protected $endPoint;
    protected $options;
    protected $access_token;
    protected $instance_url;
    protected $api;

    /**
     * SalesforceService constructor.
     * @param array $options
     */
    public function __construct(array $options = [])
    {
        $this->endPoint = 'https://login.salesforce.com/';
        $this->options = $options;
    }

    /**
     * @return bool|mixed
     * @throws \yii\base\InvalidConfigException
     * @throws \yii\httpclient\Exception
     */
    public function authenticate()
    {
        $client = new Client();

        // Send HTTP POST request
        $response = $client->createRequest()
            ->setMethod('POST')
            ->setUrl("{$this->endPoint}services/oauth2/token")
            ->addHeaders(['User-Agent' => Yii::$app->name])
            ->setData($this->options)
            ->setOptions([
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0
            ])
            ->send();

        if ($response->isOk) {
            $this->access_token = $response->data['access_token'];
            $this->instance_url = $response->data['instance_url'];

            return $response->data;
        }

        return false;
    }

    /**
     * @param $endPoint
     */
    public function setEndpoint($endPoint)
    {
        $this->endPoint = $endPoint;
    }

    /**
     * @param $accessToken
     */
    public function setAccessToken($accessToken)
    {
        $this->access_token = $accessToken;
    }

    /**
     * @return mixed
     */
    public function getAccessToken()
    {
        return $this->access_token;
    }

    /**
     * @param $instanceUrl
     */
    public function setInstanceUrl($instanceUrl)
    {
        $this->instance_url = $instanceUrl;
    }

    /**
     * @return mixed
     */
    public function getInstanceUrl()
    {
        return $this->instance_url;
    }

    /**
     * @param string $entityName Eg. 'contact'
     * @param array $values
     * @param $entityName
     * @param $values
     * @return mixed
     * @throws \Exception
     */
    public function createEntity($entityName, $values)
    {
        $api = new SalesforceApi($this->instance_url, $this->access_token);
        return $api->create($entityName, $values);
    }

    /**
     * @param string $entityName Eg. 'contact'
     * @param string $entityId
     * @param array $values
     * @return mixed
     * @throws \Exception
     */
    public function updateEntity($entityName, $entityId, $values)
    {
        $api = new SalesforceApi($this->instance_url, $this->access_token);
        return $api->update($entityName, $entityId, $values);
    }

    /**
     * Find Entity By Attribute Value
     *
     * @param string $entityName Eg. 'contact'
     * @param array $attributes Eg. ['Email' => 'john@doe.com']
     * @param array|boolean $columns Eg. ['fullname', 'Email']
     * @param $entityName
     * @param array $attributes
     * @param bool $columns
     * @return bool|mixed
     */
    public function findEntity($entityName, $attributes = [], $columns = true)
    {
        $object = $entityName;
        $columns = $columns === true ? 'ALL' : implode(',', $columns);
        $conditions = '';
        if (!empty($attributes)) {
            $counter = 0;
            foreach ($attributes as $attribute => $value) {
                if ($counter === 0) {
                    $conditions = 'WHERE ';
                }
                $conditions .= $attribute . "='" . $value . "'";
                if ($counter < count($attributes) - 1) {
                    $conditions .= ' AND ';
                }
                $counter++;
            }
        }

        $query = "SELECT {$columns} FROM {$object} {$conditions} LIMIT 1";

        return $this->findEntityByQuery($query);
    }

    /**
     * @param $query
     * @return mixed|null
     * @throws \Exception
     */
    public function findEntityByQuery($query)
    {
        $api = new SalesforceApi($this->instance_url, $this->access_token);
        $response = $api->query($query);

        $entity = null;
        if (!empty($response['records'])) {
            foreach ($response['records'] as $i => $record) {
                if ($i === 0) {
                    $entity = $record;
                }
            }
        }

        return $entity;
    }

    /**
     * @param $entityName
     * @param $entityId
     * @return bool
     * @throws \Exception
     */
    public function deleteEntity($entityName, $entityId)
    {
        $api = new SalesforceApi($this->instance_url, $this->access_token);
        return $api->delete($entityName, $entityId);
    }

    /**
     * @param $list
     * @return array
     */
    public function getListFields($list)
    {
        $api = new SalesforceApi($this->instance_url, $this->access_token);
        $response = $api->describe($list);

        $fields = [];

        if(isset($response['fields']) && is_array($response['fields'])){

            foreach ($response['fields'] as $k => $attribute) {

                if ((isset($attribute['createable']) && $attribute['createable'] == true) || $attribute['name'] == 'Id' || (isset($attribute['custom']) && $attribute['custom'] == true)) {

                    $required = "";

                    $type = $attribute['type'];

                    if ($type == 'reference' && !empty($attribute['referenceTo']) && is_array($attribute['referenceTo'])) {
                        $type = reset($attribute['referenceTo']);
                    }

                    $field = array('name' => $attribute['name'], "type" => $type);
                    $field['label'] = $attribute['label'];
                    $field["maxlength"] = $attribute['length'];
                    $field["custom"] = $attribute['custom'];

                    if (!empty($attribute['nameField']) || (!empty($attribute['createable']) && empty($attribute['nillable']) && empty($attribute['defaultedOnCreate']))) {
                        $field['required'] = "true";
                    }

                    if(isset($attribute['picklistValues']) && is_array($attribute['picklistValues']) && count($attribute['picklistValues']) > 0) {
                        $field['options'] = $attribute['picklistValues'];
                        $egs = array();
                        foreach ($attribute['picklistValues'] as $op){
                            $egs[] = $op['value'].'='.$op['label'];
                        }
                        $field['hint'] = implode(', ',array_slice($egs,0,30));
                    }

                    $fields[$attribute['name']] = $field;

                }
            }

        }

        return $fields;
    }
}