<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\salesforce\services;

use Yii;
use yii\httpclient\Client;

/**
 * Class SalesforceApi
 * @package app\modules\addons\modules\salesforce\services
 */
class SalesforceApi
{
    protected $instance_url;
    protected $access_token;
    protected $version;

    /**
     * SalesforceApi constructor.
     * @param $instance_url
     * @param $access_token
     * @param string $version
     */
    public function __construct($instance_url, $access_token, $version = 'v39.0')
    {
        $this->instance_url = $instance_url;
        $this->access_token = $access_token;
        $this->version = $version;
    }

    /**
     * @param $object
     * @return bool|mixed
     * @throws \yii\base\InvalidConfigException
     * @throws \yii\httpclient\Exception
     */
    public function describe($object)
    {
        $url = "{$this->instance_url}/services/data/{$this->version}/sobjects/{$object}/describe";

        $client = new Client();

        // Send HTTP POST request
        $response = $client->createRequest()
            ->setMethod('GET')
            ->setUrl($url)
            ->addHeaders([
                'User-Agent' => Yii::$app->name,
                'Authorization' => "OAuth {$this->access_token}",
            ])
            ->setOptions([
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0
            ])
            ->send();

        if ($response->isOk) {
            return $response->data;
        } else {
            Yii::error("Salesforce Error: " . json_encode($response->data));
        }

        return false;
    }

    /**
     * @param $query
     * @return mixed
     * @throws \yii\base\InvalidConfigException
     * @throws \yii\httpclient\Exception
     */
    public function query($query)
    {
        $url = "{$this->instance_url}/services/data/{$this->version}/query";

        $client = new Client();

        // Send HTTP POST request
        $response = $client->createRequest()
            ->setMethod('GET')
            ->setUrl($url)
            ->addHeaders([
                'User-Agent' => Yii::$app->name,
                'Authorization' => "OAuth {$this->access_token}",
            ])
            ->setData([
                'q' => $query,
            ])
            ->setOptions([
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0
            ])
            ->send();

        if (!$response->isOk) {
            $data = json_encode($response->data);
            Yii::error("Salesforce Add-On Call to URL {$url} failed with status {$response->statusCode}, data: {$data}");
            throw new \Exception("Call to Salesforce failed with status {$response->statusCode}.");
        }

        return $response->data;
    }

    /**
     * @param $object
     * @param array $data
     * @return mixed|null
     * @throws \yii\base\InvalidConfigException
     * @throws \yii\httpclient\Exception
     */
    public function create($object, array $data)
    {
        $url = "{$this->instance_url}/services/data/{$this->version}/sobjects/{$object}/";

        $client = new Client();

        // Send HTTP POST request
        $response = $client->createRequest()
            ->setMethod('POST')
            ->setUrl($url)
            ->setFormat(Client::FORMAT_JSON)
            ->addHeaders([
                'User-Agent' => Yii::$app->name,
                'Authorization' => "OAuth {$this->access_token}",
            ])
            ->setData($data)
            ->setOptions([
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0
            ])
            ->send();

        if ($response->statusCode != 201) {
            $data = json_encode($response->data);
            if (isset($response->data, $response->data[0]['errorCode'])
                && $response->data[0]['errorCode'] === 'DUPLICATES_DETECTED') {
                Yii::error("Salesforce Add-On: Duplicates detected, with status {$response->statusCode}, data: {$data}");
                return null;
            }
            Yii::error("Salesforce Add-On: Call to URL {$url} failed with status {$response->statusCode}, data: {$data}");
            throw new \Exception("Call to Salesforce failed with status {$response->statusCode}.");
        }

        $data = $response->getData();

        return $data['id'];
    }

    /**
     * @param $object
     * @param $id
     * @param array $data
     * @return mixed
     * @throws \yii\base\InvalidConfigException
     * @throws \yii\httpclient\Exception
     */
    public function update($object, $id, array $data)
    {
        $url = "{$this->instance_url}/services/data/{$this->version}/sobjects/{$object}/{$id}";

        $client = new Client();

        // Send HTTP POST request
        $response = $client->createRequest()
            ->setMethod('PATCH')
            ->setUrl($url)
            ->setFormat(Client::FORMAT_JSON)
            ->addHeaders([
                'User-Agent' => Yii::$app->name,
                'Authorization' => "OAuth {$this->access_token}",
            ])
            ->setData($data)
            ->setOptions([
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0
            ])
            ->send();

        if ($response->statusCode != 204) {
            $data = json_encode($response->data);
            Yii::error("Salesforce Add-On Call to URL {$url} failed with status {$response->statusCode}, data: {$data}");
            throw new \Exception("Call to Salesforce failed with status {$response->statusCode}.");
        }

        $data = $response->getData();

        return $data['Id'];
    }

    /**
     * @param $object
     * @param $id
     * @return bool
     * @throws \yii\base\InvalidConfigException
     * @throws \yii\httpclient\Exception
     */
    public function delete($object, $id)
    {

        $url = "{$this->instance_url}/services/data/{$this->version}/sobjects/{$object}/{$id}";

        $client = new Client();

        // Send HTTP POST request
        $response = $client->createRequest()
            ->setMethod('DELETE')
            ->setUrl($url)
            ->setFormat(Client::FORMAT_JSON)
            ->addHeaders([
                'User-Agent' => Yii::$app->name,
                'Authorization' => "OAuth {$this->access_token}",
            ])
            ->setOptions([
                CURLOPT_SSL_VERIFYHOST => 0,
                CURLOPT_SSL_VERIFYPEER => 0
            ])
            ->send();

        if ($response->statusCode != 204) {
            $data = json_encode($response->data);
            Yii::error("Salesforce Add-On Call to URL {$url} failed with status {$response->statusCode}, data: {$data}");
            throw new \Exception("Call to Salesforce failed with status {$response->statusCode}.");
        }

        return true;
    }
}