<?php

namespace app\modules\addons\modules\salesforce\controllers;

use app\components\User;
use app\helpers\Html;
use app\models\FormSubmission;
use app\modules\addons\modules\salesforce\helpers\Crm;
use app\modules\addons\modules\salesforce\models\SalesforceLog;
use app\modules\addons\modules\salesforce\models\SalesforceLogSearch;
use app\modules\addons\modules\salesforce\Module;
use Yii;
use Exception;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use app\models\Form;
use app\helpers\ArrayHelper;
use app\modules\addons\modules\salesforce\models\Salesforce;
use app\modules\addons\modules\salesforce\models\SalesforceField;
use app\modules\addons\modules\salesforce\models\SalesforceSearch;
use app\modules\addons\modules\salesforce\services\SalesforceService;
use yii\web\Response;

/**
 * AdminController implements the CRUD actions for Salesforce model.
 */
class AdminController extends Controller
{

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'list-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('Salesforce')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'send', 'sending', 'process', 'resend', 'log', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = Salesforce::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\salesforce\models\Salesforce',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'list-fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($list_id, $params) {
                    $output = array();
                    $service = null;
                    $accessToken = null;
                    $instanceUrl = null;
                    $form_id = isset($params[0]) ? $params[0] : null;
                    $model = Salesforce::findOne(['form_id' => $form_id]);
                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {

                            /** @var Module $module */
                            $module = $this->module;

                            $service = new SalesforceService([
                                'grant_type' => 'password',
                                'client_id' => $module->clientId,
                                'client_secret' => $module->clientSecret,
                                'username' => $model->username,
                                'password' => $model->password.$model->security_token,
                            ]);

                            if ($service->authenticate()) {
                                $accessToken = $service->getAccessToken();
                                $instanceUrl = $service->getInstanceUrl();
                            }
                        }
                    } else {
                        $session = Yii::$app->session;
                        $accessToken = $session['salesforce_access_token'];
                        $instanceUrl = $session['salesforce_instance_url'];
                    }
                    if ($accessToken && $instanceUrl) {
                        $service = new SalesforceService();
                        $service->setAccessToken($accessToken);
                        $service->setInstanceUrl($instanceUrl);
                        $fields = $service->getListFields($list_id);
                        usort($fields, function($a, $b) {return isset($b['required']);});
                        foreach ($fields as $field) {
                            $label = $field['label'];
                            if (!empty($field['required'])) {
                                $label = $label . ' - ' . Yii::t('app', 'Required');
                            }
                            if (!empty($field['maxlength'])) {
                                $label = $label . ' - ' . Yii::t('app', 'Max Length') . ': ' . $field['maxlength'];
                            }
                            if (!empty($field['hint'])) {
                                $label = $label . ' - ' . Yii::t('app', 'Eg.') . ' ' . $field['hint'];
                            }
                            array_push($output, [
                                'id' => $field['name'],
                                'name' => $label,
                            ]);
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($listID, $params) {
                    if (isset($params[1]) && !empty($params[1])) {
                        return $params[1];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all Salesforce models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new SalesforceSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Salesforce model.
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new Salesforce model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $lists = Crm::lists();
        $marketingLists = [];
        $connected = false;

        $request = Yii::$app->request;
        $session = Yii::$app->session;

        $model = new Salesforce();
        $fieldModel = new SalesforceField();

        try {
            if ($request->isPost && $model->load($request->post())) {

                /** @var Module $module */
                $module = $this->module;

                $service = new SalesforceService([
                    'grant_type' => 'password',
                    'client_id' => $module->clientId,
                    'client_secret' => $module->clientSecret,
                    'username' => $model->username,
                    'password' => $model->password.$model->security_token,
                ]);

                if ($service->authenticate()) {

                    $connected = true;
                    $session['salesforce_access_token'] = $service->getAccessToken();
                    $session['salesforce_instance_url'] = $service->getInstanceUrl();

                    // $marketingLists = $service->getMarketingLists();
                    $marketingLists = [];
                    $items = Yii::$app->request->post('SalesforceField',[]);

                    if (!empty($items)) {
                        if ($model->validate()) {
                            $model->items = $items;
                            $model->save(false);
                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );

                            return $this->redirect(['index']);
                        } else {
                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                            );
                        }
                    }

                } else {
                    // Show error message
                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'Please enter a valid API information.')
                    );
                }
            }
        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
            'lists' => $lists,
            'marketingLists' => $marketingLists,
            'connected' => $connected,
        ]);
    }

    /**
     * Updates an existing Salesforce model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param $id
     * @return string|\yii\web\Response
     * @throws Exception
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        $request = Yii::$app->request;
        $session = Yii::$app->session;
        $connected = false;
        $lists = Crm::lists();
        $marketingLists = [];

        try {

            /** @var Module $module */
            $module = $this->module;

            $service = new SalesforceService([
                'grant_type' => 'password',
                'client_id' => $module->clientId,
                'client_secret' => $module->clientSecret,
                'username' => $model->username,
                'password' => $model->password.$model->security_token,
            ]);

            if ($service->authenticate()) {

                $connected = true;
                $session['salesforce_access_token'] = $service->getAccessToken();
                $session['salesforce_instance_url'] = $service->getInstanceUrl();
                // $marketingLists = $service->getMarketingLists();

                if ($request->isPost && $model->load($request->post())) {

                    $items = Yii::$app->request->post('SalesforceField',[]);

                    if (!empty($items)) {

                        if ($model->validate()) {

                            $model->items = $items;
                            $model->save(false);

                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );

                            return $this->redirect(['index']);

                        } else {

                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                            );
                        }
                    }
                }

            } else {
                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    Yii::t('app', 'Please enter a valid API information.')
                );
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'fieldModel' => new SalesforceField,
            'forms' => $forms,
            'lists' => $lists,
            'marketingLists' => $marketingLists,
            'connected' => $connected,
        ]);
    }

    /**
     * Enable / Disable multiple CleverReach models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = Salesforce::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing Salesforce model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Send form submissions to Salesforce.
     * If configuration is successful, the browser will be redirected to the 'sending' page.
     *
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionSend($id)
    {

        $model = $this->findModel($id);
        $request = Yii::$app->request;
        $session = Yii::$app->session;
        $connected = false;

        try {

            /** @var Module $module */
            $module = $this->module;

            $service = new SalesforceService([
                'grant_type' => 'password',
                'client_id' => $module->clientId,
                'client_secret' => $module->clientSecret,
                'username' => $model->username,
                'password' => $model->password.$model->security_token,
            ]);

            if ($service->authenticate()) {

                $connected = true;
                $session['salesforce_access_token'] = $service->getAccessToken();
                $session['salesforce_instance_url'] = $service->getInstanceUrl();

                if ($request->isPost) {

                    if (Yii::$app->request->post('resent', false)) {

                        return $this->redirect(['sending', 'id' => $model->id, 'resent' => 1]);

                    }

                    return $this->redirect(['sending', 'id' => $model->id]);

                }

            } else {
                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                );

                return $this->redirect(['index']);
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('send', [
            'model' => $model,
            'forms' => $forms,
            'connected' => $connected,
        ]);
    }

    /**
     * UI to show the progress bar while sending form submissions to Salesforce.
     * It shows a progress bar.
     *
     * @param $id
     * @param $resent
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionSending($id, $resent = 0)
    {
        $model = $this->findModel($id);

        return $this->render('sending', [
            'model' => $model,
            'resent' => $resent,
        ]);
    }

    /**
     * Submit form submissions to the Salesforce using a batch process
     * @param int $id
     * @param int $resent
     * @param int $step
     * @param int $limit
     * @return array
     */
    public function actionProcess($id, $resent = 0, $step = 0, $limit = 1)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        try {

            $model = $this->findModel($id);

            if ($model->status === Salesforce::ON) {

                // Set number of rows
                $nbRows = FormSubmission::find()
                    ->where(['form_id' => $model->form_id])
                    ->count();

                // Set offset
                $offset = $step * $limit;

                // Batch percentage
                $percentage = round(($offset + $limit) * 100 / $nbRows);

                // All rows have been processed ?
                $completed = ($offset + $limit) >= $nbRows;

                $rows = FormSubmission::find()
                    ->where(['form_id' => $model->form_id])
                    ->offset($offset)
                    ->limit($limit)
                    ->orderBy(['id' => SORT_DESC])
                    ->all();

                // Batch
                /** @var FormSubmission $row */
                foreach ($rows as $row) {

                    // Avoid sending previously sent entries
                    if (!$resent) {

                        $sent = SalesforceLog::findOne([
                            'salesforce_id' => $model->id,
                            'submission_id' => $row->id,
                            'status' => 1,
                        ]);

                        if ($sent) {
                            continue;
                        }
                    }

                    /** @var Module $module */
                    $module = $this->module;

                    try {

                        // Submit data to Salesforce according to the model configuration
                        $result = $module->submitData($model, $row->getSubmissionData());
                        // And log result
                        $module->logResult($model, $row, $result);

                    } catch (\Exception $e) {

                        // Log error
                        Yii::error($e);

                    }

                }

                $message = Yii::t('app',
                    'We have processed a batch of {0} form submissions.', [
                        Html::tag('span', count($rows), ['style' => 'font-weight: bold']),
                    ]);

                if ($completed) {
                    $message = Yii::t('app',
                        'Ready! We have processed {0} form submissions.', [
                            Html::tag('span', $nbRows, ['style' => 'font-weight: bold']),
                        ]);
                }

                return [
                    'completed' => $completed,
                    'step' => $completed ? 'done' : $step + 1,
                    'percentage' => $completed ? 100 : $percentage,
                    'limit' => $limit,
                    'message' => $message,
                ];

            }

        } catch (Exception $e) {
            Yii::error($e);
            return [
                'error' => 1,
            ];
        }

        return [
            'error' => 2,
        ];
    }

    /**
     * Re-Send a form submission to Salesforce.
     * If configuration is successful, the browser will be redirected to the 'previous' page.
     *
     * @param int $id Salesforce ID
     * @param int $sid Submission ID
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionResend($id, $sid)
    {

        $model = $this->findModel($id);

        try {

            $submission = FormSubmission::findOne(['id' => $sid]);

            // Verify if Submission ID was collected by the Form linked to this Salesforce ID
            if ($model->form_id === $submission->form_id) {

                /** @var Module $module */
                $module = $this->module;

                $service = new SalesforceService([
                    'grant_type' => 'password',
                    'client_id' => $module->clientId,
                    'client_secret' => $module->clientSecret,
                    'username' => $model->username,
                    'password' => $model->password.$model->security_token,
                ]);

                if ($service->authenticate()) {

                    $result = $module->submitData($model, $submission->getSubmissionData());
                    // And log result
                    $module->logResult($model, $submission, $result);

                    if ($result) {

                        Yii::$app->getSession()->setFlash(
                            'success',
                            Yii::t('app','Form Submission was successfully re-sent to Salesforce.')
                        );

                    } else {

                        Yii::$app->getSession()->setFlash(
                            'danger',
                            Yii::t('app', 'An error occurred re-sending the form submission to Salesforce.')
                        );

                    }

                } else {

                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'Invalid Salesforce settings found. Please verify your configuration.')
                    );
                }
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        $referrer = !empty(Yii::$app->request->referrer) ? Yii::$app->request->referrer : null;

        return $this->goBack($referrer);
    }

    /**
     * Lists all SalesforceLog models.
     *
     * @param $id
     *
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionLog($id)
    {
        $searchModel = new SalesforceLogSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('log', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Finds the Salesforce model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Salesforce the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Salesforce::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
