<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\salesforce;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\salesforce\models\Salesforce;
use app\modules\addons\modules\salesforce\models\SalesforceField;
use app\modules\addons\modules\salesforce\models\SalesforceLog;
use app\modules\addons\modules\salesforce\services\SalesforceService;
use Exception;
use Yii;
use yii\base\View;
use yii\helpers\Url;

/**
 * Class Module
 * @package app\modules\addons\modules\salesforce
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "salesforce";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';
    public $clientId = '3MVG9ZF4bs_.MKugy8D.U5CC1FNOTRVEEXCzfC_ma1q9VnEZpSaIFuQWsHHoFa6YWx3SR5VTVItSc1rl7f9lm';
    public $clientSecret = '210BDEF18C69E3E68C6B2EE7D6F889CED4073EA8391C33D8EBE1B08FD83BA49A';
    public static $rendered = false;

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'Salesforce' => Salesforce::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            View::class => [
                'afterRender' => [
                    [Module::class, 'onViewAfterRender']
                ],
            ],
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = Salesforce::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new Salesforce();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new SalesforceField();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->salesforce_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = Salesforce::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "form/submissions"
        ) {
            $formID = $event->sender->context->request->get("id");
            $formModel = Form::findOne(['id' => $formID]);

            // Flag to improve performance
            if (!self::$rendered && Yii::$app->user->can('configureFormsWithAddons', ['model' => $formModel])) {

                $models = Salesforce::findAll(['form_id' => $formModel->id, 'status' => 1]);

                $content = $event->output;

                foreach ($models as $model) {
                    $content = str_replace(
                        '<i class="glyphicon glyphicon-print"></i>',
                        '<i class="glyphicon glyphicon-print"></i></a><a class="btn btn-info btn-salesforce" href="#" data-href="' . Url::to(['/addons/salesforce/admin/resend', 'id' => $model->id], true) . '" title="'.Yii::t('app', 'Send Form Submission to Salesforce') . '"> <i class="glyphicon glyphicon-send"></i> ',
                        $content);
                }

                $content .= <<<EOT
<script>
window.addEventListener('load', function(){
  $('body').on('click', '.btn-salesforce', function(event) {
        event.preventDefault();
        var hash = location.hash.substr(1),
        submission_id = hash.replace(/[^0-9]/g, ''),
        redirectWindow = window.open($(this).data('href') + '&sid=' + submission_id, '_blank');
        redirectWindow.location;
  });
});
</script>
EOT;

                $event->output =  $content;

                self::$rendered = true;
            }
        }
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {
        $result = false;

        $models = Salesforce::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var \app\models\FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                try {

                    // Submit data to Salesforce according to the model configuration
                    $result = $this->submitData($model, $submissionData);
                    // Log result
                    $this->logResult($model, $submissionModel, $result);

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;
    }

    /**
     * Submit form submission to Salesforce
     *
     * @param Salesforce $model Salesforce model
     * @param array $data Submission Data
     * @return mixed|null
     * @throws \Exception
     */
    public function submitData($model, $data)
    {

        // Flag
        $entityID = null;

        $values = [];

        // Validate, format and set Field values
        foreach ($model->items as $item) {
            if (!empty($data[$item->form_field])) {
                // Default value
                $values[$item->list_field] = $data[$item->form_field];

                // Select List, Checkbox
                if (is_array($data[$item->form_field])) {
                    if (count($data[$item->form_field]) > 1) {
                        $values[$item->list_field] = implode(',', $data[$item->form_field]);
                    } else {
                        $values[$item->list_field] = $data[$item->form_field][0];
                    }
                }

                // Numbers
                if (is_numeric($data[$item->form_field])) {
                    if ((int)$data[$item->form_field] == (float)$data[$item->form_field]) {
                        $values[$item->list_field] = (int)$data[$item->form_field];
                    } else {
                        $values[$item->list_field] = (float)$data[$item->form_field];
                    }
                }
            }
        }

        if (!empty($values)) {

            $service = new SalesforceService([
                'grant_type' => 'password',
                'client_id' => $this->clientId,
                'client_secret' => $this->clientSecret,
                'username' => $model->username,
                'password' => $model->password.$model->security_token,
            ]);

            if ($service->authenticate()) {

                $customerEntity = null;
                $entity = null;
                $entityType = is_array($model->list_id) ? current($model->list_id) : $model->list_id;

                /**
                 * Search Customer Entity (Contact or Lead)
                 */
                $priorities = $model->search_priority === Salesforce::SEARCH_PRIORITY_BY_CONTACT ? ['Contact', 'Lead'] : ['Lead', 'Contact'];
                if (!empty($data[$model->email_field])) {
                    foreach ($priorities as $entityTypeWithPriority) {
                        $columns = $entityTypeWithPriority === 'Contact' ? ['Id', 'AccountId'] : ['Id'];
                        $customerEntity = $service->findEntity($entityTypeWithPriority, [
                            $model->search_by => $data[$model->email_field]
                        ], $columns);
                        if ($customerEntity && !empty($customerEntity['Id'])) {
                            break;
                        }
                    }
                }

                /**
                 * Save entity
                 */

                if ($model->action === Salesforce::ACTION_SEARCH_AND_CREATE) {

                    if ($entityType === 'Lead') {
                        $entityID = $service->createEntity($entityType, $values);
                    } elseif ($entityType === 'Contact') {
                        $entityID = $service->createEntity($entityType, $values);
                    } elseif ($entityType === 'Account') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {
                            // Create account and link contact
                            $entityID = $service->createEntity($entityType, $values);
                            $service->updateEntity('Contact', $customerEntity['Id'], [
                                'AccountId' => $entityID,
                            ]);
                        } else {
                            // Only create
                            $entityID = $service->createEntity($entityType, $values);
                        }

                    } elseif ($entityType === 'Opportunity') {

                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {

                            if (!empty($customerEntity["AccountId"])) {
                                // Link account to opportunity
                                $values['AccountId'] = $customerEntity["AccountId"];
                            } elseif (!empty($data[$model->account_name_field])) {
                                // Create account and link contact
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);
                                $service->updateEntity('Contact', $customerEntity['Id'], [
                                    'AccountId' => $accountID,
                                ]);
                                // Link account to opportunity
                                $values['AccountId'] = $accountID;
                            }

                            // Create opportunity
                            $entityID = $service->createEntity($entityType, $values);

                        } else {

                            if (!empty($data[$model->account_name_field])) {
                                // Create account
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);

                                // Create contact
                                if (!empty($data[$model->first_name_field])
                                    && !empty($data[$model->last_name_field])
                                    && !empty($data[$model->email_field])) {
                                    $contactID = $service->createEntity('Contact', [
                                        'FirstName' => $data[$model->first_name_field],
                                        'LastName' => $data[$model->last_name_field],
                                        'Email' => $data[$model->email_field],
                                        'AccountId' => $accountID,
                                    ]);
                                }

                                // Link account to opportunity
                                $values['AccountId'] = $accountID;
                            }

                            // Create opportunity
                            $entityID = $service->createEntity($entityType, $values);
                        }

                    } elseif ($entityType === 'Case') {

                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {

                            if (!empty($customerEntity["AccountId"])) {
                                // Link account to case
                                $values['AccountId'] = $customerEntity["AccountId"];
                            } elseif (!empty($data[$model->account_name_field])) {
                                // Create account and link contact
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);
                                $service->updateEntity('Contact', $customerEntity['Id'], [
                                    'AccountId' => $accountID,
                                ]);
                                // Link account to case
                                $values['AccountId'] = $accountID;
                            }

                            // Link contact to case
                            $values['ContactId'] = $customerEntity["Id"];

                            // Create case
                            $entityID = $service->createEntity($entityType, $values);

                        } else {

                            // Create account
                            if (!empty($data[$model->account_name_field])) {
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);

                                // Link account to case
                                $values['AccountId'] = $accountID;
                            }

                            // Create contact
                            if (!empty($data[$model->first_name_field])
                                && !empty($data[$model->last_name_field])
                                && !empty($data[$model->email_field])) {
                                $attr = [
                                    'FirstName' => $data[$model->first_name_field],
                                    'LastName' => $data[$model->last_name_field],
                                    'Email' => $data[$model->email_field],
                                ];
                                if (!empty($accountID)) {
                                    $attr['AccountId'] = $accountID;
                                }
                                $contactID = $service->createEntity('Contact', $attr);
                                // Link contact to case
                                $values['ContactId'] = $contactID;
                            }

                            // Create case
                            $entityID = $service->createEntity($entityType, $values);
                        }

                    }

                } elseif ($model->action === Salesforce::ACTION_SEARCH_AND_UPDATE) {

                    if ($entityType === 'Lead') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Lead') {
                            $service->updateEntity($entityType, $customerEntity['Id'], $values);
                            $entityID = $customerEntity['Id'];
                        }
                    } elseif ($entityType === 'Contact') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {
                            $service->updateEntity($entityType, $customerEntity['Id'], $values);
                            $entityID = $customerEntity['Id'];
                        }
                    } elseif ($entityType === 'Account') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {
                            // Update Account by Contact relationship
                            if (!empty($customerEntity['AccountId'])) {
                                $service->updateEntity('Account', $customerEntity['AccountId'], $values);
                                $entityID = $customerEntity['AccountId'];
                            }
                        }
                    } elseif ($entityType === 'Opportunity') {
                        // We can't update an opportunity
                    } elseif ($entityType === 'Case') {
                        // We can't update an incident
                    }

                } elseif ($model->action === Salesforce::ACTION_SEARCH_AND_UPDATE_OR_CREATE) {

                    if ($entityType === 'Lead') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Lead') {
                            $service->updateEntity($entityType, $customerEntity['Id'], $values);
                            $entityID = $customerEntity['Id'];
                        } else {
                            $entityID = $service->createEntity($entityType, $values);
                        }
                    } elseif ($entityType === 'Contact') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {
                            $service->updateEntity($entityType, $customerEntity['Id'], $values);
                            $entityID = $customerEntity['Id'];
                        } else {
                            $entityID = $service->createEntity($entityType, $values);
                        }
                    } elseif ($entityType === 'Account') {
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {
                            if (!empty($customerEntity['AccountId'])) {
                                // Update Contact Account
                                $service->updateEntity('Account', $customerEntity['AccountId'], $values);
                                $entityID = $customerEntity['AccountId'];
                            } else {
                                // Create account and link contact
                                $entityID = $service->createEntity($entityType, $values);
                                $service->updateEntity('Contact', $customerEntity['Id'], [
                                    'AccountId' => $entityID,
                                ]);
                            }
                        } else {
                            // Only create
                            $entityID = $service->createEntity($entityType, $values);
                        }
                    } elseif ($entityType === 'Opportunity') {

                        // We can't update an opportunity, only create one
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {

                            if (!empty($customerEntity["AccountId"])) {
                                // Link account to opportunity
                                $values['AccountId'] = $customerEntity["AccountId"];
                            } elseif (!empty($data[$model->account_name_field])) {
                                // Create account and link contact
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);
                                $service->updateEntity('Contact', $customerEntity['Id'], [
                                    'AccountId' => $accountID,
                                ]);
                                // Link account to opportunity
                                $values['AccountId'] = $accountID;
                            }

                            // Create opportunity
                            $entityID = $service->createEntity($entityType, $values);

                        } else {

                            if (!empty($data[$model->account_name_field])) {
                                // Create account
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);

                                // Create contact
                                if (!empty($data[$model->first_name_field])
                                    && !empty($data[$model->last_name_field])
                                    && !empty($data[$model->email_field])) {
                                    $contactID = $service->createEntity('Contact', [
                                        'FirstName' => $data[$model->first_name_field],
                                        'LastName' => $data[$model->last_name_field],
                                        'Email' => $data[$model->email_field],
                                        'AccountId' => $accountID,
                                    ]);
                                }

                                // Link account to opportunity
                                $values['AccountId'] = $accountID;
                            }

                            // Create opportunity
                            $entityID = $service->createEntity($entityType, $values);
                        }

                    } elseif ($entityType === 'Case') {

                        // We can't update an incident, only create one
                        if ($customerEntity
                            && !empty($customerEntity["attributes"]["type"])
                            && $customerEntity["attributes"]["type"] === 'Contact') {

                            if (!empty($customerEntity["AccountId"])) {
                                // Link account to case
                                $values['AccountId'] = $customerEntity["AccountId"];
                            } elseif (!empty($data[$model->account_name_field])) {
                                // Create account and link contact
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);
                                $service->updateEntity('Contact', $customerEntity['Id'], [
                                    'AccountId' => $accountID,
                                ]);
                                // Link account to case
                                $values['AccountId'] = $accountID;
                            }

                            // Link contact to case
                            $values['ContactId'] = $customerEntity["Id"];

                            // Create case
                            $entityID = $service->createEntity($entityType, $values);

                        } else {

                            // Create account
                            if (!empty($data[$model->account_name_field])) {
                                $accountID = $service->createEntity('Account', [
                                    'Name' => $data[$model->account_name_field],
                                ]);

                                // Link account to case
                                $values['AccountId'] = $accountID;
                            }

                            // Create contact
                            if (!empty($data[$model->first_name_field])
                                && !empty($data[$model->last_name_field])
                                && !empty($data[$model->email_field])) {
                                $attr = [
                                    'FirstName' => $data[$model->first_name_field],
                                    'LastName' => $data[$model->last_name_field],
                                    'Email' => $data[$model->email_field],
                                ];
                                if (!empty($accountID)) {
                                    $attr['AccountId'] = $accountID;
                                }
                                $contactID = $service->createEntity('Contact', $attr);
                                // Link contact to case
                                $values['ContactId'] = $contactID;
                            }

                            // Create case
                            $entityID = $service->createEntity($entityType, $values);
                        }
                    }
                }

                /**
                 * Qualify lead
                 */

                if (!empty($entityID) && $model->qualify_lead === Salesforce::ON) {
                    if ($entityType === 'Lead') {

                        $lead = $service->findEntity('Lead', ['Id' => $entityID], ['Id', 'Status']);

                        if ($lead
                            && !empty($lead['Id'])
                            && !empty($lead['Status'])
                            && in_array($lead['Status'], ['Open - Not Contacted', 'Working - Contacted'])) {

                            // Update lead
                            $service->updateEntity('Lead', $lead['Id'], [
                                'Status' => 'Closed - Converted'
                            ]);

                            // Reset values
                            $accountID = null;
                            $contactID = null;

                            if ($model->create_account === Salesforce::ON) {
                                // Create account
                                if (!empty($data[$model->account_name_field])) {
                                    $account = $service->findEntity('Account', ['Name' => $data[$model->account_name_field]], ['Id', 'Name']);
                                    if ($account && !empty($account['Id'])) {
                                        $accountID = $account['Id'];
                                    } else {
                                        $accountID = $service->createEntity('Account', [
                                            'Name' => $data[$model->account_name_field],
                                        ]);
                                    }
                                }
                            }

                            if ($model->create_opportunity === Salesforce::ON) {
                                // Create opportunity
                                $attr = [
                                    'Name' => Yii::t('app', 'New opportunity'),
                                    'StageName' => 'Prospecting',
                                    'CloseDate' => date('Y-m-d', strtotime('+1 month', time())),
                                ];
                                if (!empty($accountID)) {
                                    $attr['AccountId'] = $accountID;
                                }
                                $service->createEntity('Opportunity', $attr);
                            }

                            if ($model->create_contact === Salesforce::ON) {
                                $contact = $service->findEntity('Contact', ['Email' => $data[$model->email_field]], ['Id', 'Name']);
                                // Create contact
                                if (empty($contact['Id'])) {
                                    if (!empty($data[$model->first_name_field])
                                        && !empty($data[$model->last_name_field])
                                        && !empty($data[$model->email_field])) {

                                        $attr = [
                                            'FirstName' => $data[$model->first_name_field],
                                            'LastName' => $data[$model->last_name_field],
                                            'Email' => $data[$model->email_field],
                                        ];
                                        if (!empty($accountID)) {
                                            $attr['AccountId'] = $accountID;
                                        }
                                        $service->createEntity('Contact', $attr);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        return $entityID;
    }

    /**
     * @param Salesforce $salesforceModel
     * @param FormSubmission $submissionModel
     * @param string $result
     */
    public function logResult($salesforceModel, $submissionModel, $result)
    {
        // Update / Insert Log
        $log = SalesforceLog::findOne([
            'salesforce_id' => $salesforceModel->id,
            'form_id' => $salesforceModel->form_id,
            'submission_id' => $submissionModel->id]);

        if (!$log) {
            $log = new SalesforceLog();
        }

        $log->salesforce_id = $salesforceModel->id;
        $log->form_id = $salesforceModel->form_id;
        $log->submission_id = $submissionModel->id;
        $log->log = json_encode(['entity_id' => $result]);
        $log->status = !empty($result);
        $log->save(false);
    }
}