<?php

namespace app\modules\addons\modules\pdf\services;

use Exception;
use Monolog\Handler\IFTTTHandler;
use Yii;

class PdfService
{
    /**
     * PDF Certificate Path
     */
    const PDF_CERTIFICATE = 'file://pdf.crt';
    /**
     * PDF Signature Image Path
     */
    const PDF_SIGNATURE = '@app/pdf_signature.png';
    /**
     * PDF Signature Appearance
     * X (float) Abscissa of the upper-left corner.
     * Y (float) Ordinate of the upper-left corner.
     * W (float) Width of the signature area.
     * Y (float) Height of the signature area.
     */
    const PDF_SIGNATURE_X = 10;
    const PDF_SIGNATURE_Y = 10;
    const PDF_SIGNATURE_W = 20;
    const PDF_SIGNATURE_H = 20;
    /**
     * File output sent to browser inline
     */
    const DEST_BROWSER = 'I';
    /**
     * File output sent for direct download
     */
    const DEST_DOWNLOAD = 'D';
    /**
     * File output sent to a file
     */
    const DEST_FILE = 'F';
    /**
     * File output sent as a string
     */
    const DEST_STRING = 'S';

    /** @var $html */
    protected $html;

    /** @var $htmlFile */
    protected $htmlFile;

    /** @var string $size */
    protected $size = 'A4';

    /** @var string $orientation */
    protected $orientation = 'portrait';

    /** @var int $dpi */
    protected $dpi = 72;

    /** @var bool $isHtml5ParserEnabled */
    protected $isHtml5ParserEnabled = true;

    /** @var bool $isRemoteEnabled */
    protected $isRemoteEnabled = true;

    /** @var bool $isJavascriptEnabled */
    protected $isJavascriptEnabled = true;

    /** @var bool $isPhpEnabled */
    protected $isPhpEnabled = false;

    /** @var string $filename */
    protected $filename = 'document.pdf';

    /** @var $destination */
    protected $destination;

    /** @var $userPassword */
    protected $userPassword;

    /** @var $adminPassword */
    protected $adminPassword;

    /** @var bool $isSignatureEnabled */
    protected $isSignatureEnabled = false;

    /** @var \Dompdf\Dompdf $dompdf */
    protected $dompdf;

    /** @var \Mpdf\Mpdf $mpdf */
    protected $mpdf;

    /** @var \setasign\Fpdi\Tcpdf\Fpdi $mpdf */
    protected $fpdi;

    /** @var string $privateKeyPassword */
    protected $privateKeyPassword;

    /**
     * PdfService constructor.
     */
    public function __construct()
    {

        require_once(Yii::getAlias('@addons/pdf/services/vendor/autoload.php'));

        if (class_exists(\Dompdf\Dompdf::class)) {
            $this->dompdf = new \Dompdf\Dompdf();
        }

        if (class_exists(\Mpdf\Mpdf::class)) {
            $this->mpdf = new \Mpdf\Mpdf();
        }

        if (class_exists(\setasign\Fpdi\Tcpdf\Fpdi::class)
        && class_exists(\TCPDF::class)) {
            $this->fpdi = new \setasign\Fpdi\Tcpdf\Fpdi();
            $this->privateKeyPassword = Yii::$app->params['App.Addons.PDF.privateKeyPassword'] ?? '';
        }

    }

    /**
     * @return mixed
     */
    public function getHtml()
    {
        return $this->html;
    }

    /**
     * @param mixed $html
     */
    public function setHtml($html)
    {
        $this->html = $html;
    }

    /**
     * @return mixed
     */
    public function getHtmlFile()
    {
        return $this->htmlFile;
    }

    /**
     * @param mixed $htmlFile
     */
    public function setHtmlFile($htmlFile)
    {
        $this->htmlFile = $htmlFile;
    }

    /**
     * @return mixed
     */
    public function getSize()
    {
        return $this->size;
    }

    /**
     * @param mixed $size
     */
    public function setSize($size)
    {
        $this->size = $size ? $size : $this->size;
    }

    /**
     * @return mixed
     */
    public function getOrientation()
    {
        if ($this->orientation === 'L') {
            return 'landscape';
        }

        return 'portrait';
    }

    /**
     * @param mixed $orientation
     */
    public function setOrientation($orientation)
    {
        $this->orientation = $orientation ? $orientation : $this->orientation;
    }

    /**
     * @return array
     */
    public function getPaperSizes()
    {
        $sizes = [];

        if ($this->dompdf) {
            foreach (\Dompdf\Adapter\CPDF::$PAPER_SIZES as $label => $size) {
                $sizes[$label] = ucwords($label);
            }
        }

        return $sizes;
    }

    /**
     * @return bool
     */
    public function isHtml5ParserEnabled()
    {
        return $this->isHtml5ParserEnabled;
    }

    /**
     * @param bool $isHtml5ParserEnabled
     */
    public function setIsHtml5ParserEnabled($isHtml5ParserEnabled)
    {
        $this->isHtml5ParserEnabled = $isHtml5ParserEnabled;
    }

    /**
     * @return bool
     */
    public function isRemoteEnabled()
    {
        return $this->isRemoteEnabled;
    }

    /**
     * @param bool $isRemoteEnabled
     */
    public function setIsRemoteEnabled($isRemoteEnabled)
    {
        $this->isRemoteEnabled = $isRemoteEnabled;
    }

    /**
     * @return bool
     */
    public function isJavascriptEnabled()
    {
        return $this->isJavascriptEnabled;
    }

    /**
     * @param bool $isJavascriptEnabled
     */
    public function setIsJavascriptEnabled($isJavascriptEnabled)
    {
        $this->isJavascriptEnabled = $isJavascriptEnabled;
    }

    /**
     * @return bool
     */
    public function isPhpEnabled()
    {
        return $this->isPhpEnabled;
    }

    /**
     * @param bool $isPhpEnabled
     */
    public function setIsPhpEnabled($isPhpEnabled)
    {
        $this->isPhpEnabled = $isPhpEnabled;
    }

    /**
     * @return mixed
     */
    public function getDpi()
    {
        return $this->dpi;
    }

    /**
     * @param mixed $dpi
     */
    public function setDpi($dpi)
    {
        $this->dpi = $dpi ? $dpi : $this->dpi;
    }

    /**
     * @return string
     */
    public function getFilename()
    {
        return $this->filename;
    }

    /**
     * @param string $filename
     */
    public function setFilename($filename)
    {
        $this->filename = !empty($filename) ? $filename : $this->filename;
    }

    /**
     * @return string
     */
    public function getDestination()
    {
        return $this->destination;
    }

    /**
     * @param string $destination
     */
    public function setDestination($destination)
    {
        $this->destination = $destination ? $destination : $this->destination;
    }

    /**
     * @return mixed
     */
    public function getUserPassword()
    {
        return $this->userPassword;
    }

    /**
     * @param mixed $userPassword
     */
    public function setUserPassword($userPassword)
    {
        $this->userPassword = $userPassword;
    }

    /**
     * @return mixed
     */
    public function getAdminPassword()
    {
        return $this->adminPassword;
    }

    /**
     * @param mixed $adminPassword
     */
    public function setAdminPassword($adminPassword)
    {
        $this->adminPassword = $adminPassword;
    }

    /**
     * @return mixed
     */
    public function digitallySign()
    {
        return $this->isSignatureEnabled
            && @file_exists(Yii::getAlias(str_replace('file:/', '@app', self::PDF_CERTIFICATE)))
            && $this->fpdi;
    }

    /**
     * @param mixed $size
     */
    public function setIsSignatureEnabled($isSignatureEnabled)
    {
        $this->isSignatureEnabled = (bool) $isSignatureEnabled;
    }

    /**
     * @param string $destination
     * @return bool|int
     * @throws \Exception
     */
    public function save($destination = self::DEST_BROWSER)
    {
        if ($this->mpdf) {

            $this->mpdf = new \Mpdf\Mpdf([
                'mode' => '',
                'format' => $this->getSize(),
                'default_font_size' => 0,
                'default_font' => '',
                'margin_left' => 15,
                'margin_right' => 15,
                'margin_top' => 16,
                'margin_bottom' => 16,
                'margin_header' => 9,
                'margin_footer' => 9,
                'orientation' => $this->getOrientation(),
            ]);
            $this->mpdf->autoScriptToLang = true;
            $this->mpdf->autoLangToFont = true;

            // Security
            if (!empty($this->adminPassword) && !empty($this->userPassword)) {
                $this->mpdf->SetProtection(array(), $this->getUserPassword(), $this->getAdminPassword());
            }

            // Save File
            $filename = $this->getFilename();
            $destination = $this->destination ? $this->destination : $destination;

            $html = $this->getHtml();
            $this->mpdf->WriteHTML($html);

            if ($destination === self::DEST_BROWSER) {
                return $this->mpdf->Output();
            } elseif ($destination === self::DEST_DOWNLOAD) {
                // For file downloads
                if ($this->digitallySign()) {
                    // Save file with mPDF
                    $this->mpdf->output(Yii::getAlias("@app/$filename"), self::DEST_FILE);
                    // Overwrite (Sign) with FPDI & TCPDF
                    $fpdi = $this->fpdi;
                    $pageCount = $fpdi->setSourceFile($filename);
                    for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
                        // remove default header/footer
                        $fpdi->SetMargins(0, 0, 0);
                        $fpdi->setPrintHeader(false);
                        $fpdi->setPrintFooter(false);
                        $templateId = $fpdi->importPage($pageNo);
                        $fpdi->AddPage();
                        $fpdi->useTemplate($templateId, 0, 0, null, null, true);
                        // Print Signature Image
                        if (@file_exists(Yii::getAlias(self::PDF_SIGNATURE)) && $pageNo === 1) {
                            $fpdi->Image(Yii::getAlias(self::PDF_SIGNATURE),self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H,'PNG');
                            $fpdi->setSignatureAppearance(self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H);
                        }
                    }
                    $fpdi->setSignature(self::PDF_CERTIFICATE, self::PDF_CERTIFICATE, $this->privateKeyPassword, '', 2, []);
                    $fpdi->Output(basename($filename), $destination);
                    die;
                }
                $this->mpdf->Output(basename($filename), $destination);
                die;
            } elseif ($destination === self::DEST_STRING) {
                return $this->mpdf->output($filename, $destination);
            }

            // For email attachment
            if ($this->digitallySign()) {
                // Save file with mPDF
                if ($this->mpdf->output($filename, $destination) === NULL) {
                    // Overwrite (Sign) with FPDI & TCPDF
                    $fpdi = $this->fpdi;
                    $pageCount = $fpdi->setSourceFile(Yii::getAlias("@app/$filename"));
                    for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
                        // remove default header/footer
                        $fpdi->SetMargins(0, 0, 0);
                        $fpdi->setPrintHeader(false);
                        $fpdi->setPrintFooter(false);
                        $templateId = $fpdi->importPage($pageNo);
                        $fpdi->AddPage();
                        $fpdi->useTemplate($templateId, 0, 0, null, null, true);
                        // Print Signature Image
                        if (@file_exists(Yii::getAlias(self::PDF_SIGNATURE)) && $pageNo === 1) {
                            $fpdi->Image(Yii::getAlias(self::PDF_SIGNATURE),self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H,'PNG');
                            $fpdi->setSignatureAppearance(self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H);
                        }
                    }
                    $fpdi->setSignature(self::PDF_CERTIFICATE, self::PDF_CERTIFICATE, $this->privateKeyPassword, '', 2, []);
                    return $fpdi->Output(Yii::getAlias("@app/$filename"), $destination) === '';
                }
            }

            return $this->mpdf->output($filename, $destination) === NULL;

        } elseif ($this->dompdf) {

            // Options
            $options = new \Dompdf\Options();
            $options->set('isHtml5ParserEnabled', $this->isHtml5ParserEnabled());
            $options->set('isPhpEnabled', $this->isPhpEnabled());
            $options->set('isRemoteEnabled', $this->isRemoteEnabled());
            $options->set('isJavascriptEnabled', $this->isJavascriptEnabled());
            $options->set('dpi', $this->getDpi());
            $this->dompdf = new \Dompdf\Dompdf($options);

            // Size and Orientation
            $this->dompdf->setPaper($this->getSize(), $this->getOrientation());

            // HTML Content
            if ($this->htmlFile) {
                $this->dompdf->loadHtmlFile($this->htmlFile);
            } else {
                $html = $this->getHtml();
                if ($html) {
                    // Fix know issues (Division by zero)
                    $html = str_replace('cellspacing="0"', '', $html);
                    $html = str_replace('cellpadding="0"', '', $html);
                }
                $this->dompdf->loadHtml($html);
            }

            // Render
            $this->dompdf->render();

            // Security
            if (!empty($this->adminPassword) && !empty($this->userPassword)) {
                $this->dompdf->getCanvas()->get_cpdf()->setEncryption($this->getUserPassword(), $this->getAdminPassword());
            }

            // Save File
            $filename = $this->getFilename();
            $destination = $this->destination ? $this->destination : $destination;
            if ($destination === self::DEST_BROWSER) {
                $this->dompdf->stream($filename, array('Attachment' => 0));
                die;
            } elseif ($destination === self::DEST_DOWNLOAD) {
                // For file downloads
                if ($this->digitallySign()) {
                    // Save file with Dompdf
                    file_put_contents($filename, $this->dompdf->output());
                    // Overwrite (Sign) with FPDI & TCPDF
                    $fpdi = $this->fpdi;
                    $pageCount = $fpdi->setSourceFile($filename);
                    for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
                        // remove default header/footer
                        $fpdi->SetMargins(0, 0, 0);
                        $fpdi->setPrintHeader(false);
                        $fpdi->setPrintFooter(false);
                        $templateId = $fpdi->importPage($pageNo);
                        $fpdi->AddPage();
                        $fpdi->useTemplate($templateId, 0, 0, null, null, true);
                        // Print Signature Image
                        if (@file_exists(Yii::getAlias(self::PDF_SIGNATURE)) && $pageNo === 1) {
                            $fpdi->Image(Yii::getAlias(self::PDF_SIGNATURE),self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H,'PNG');
                            $fpdi->setSignatureAppearance(self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H);
                        }
                    }
                    $fpdi->setSignature(self::PDF_CERTIFICATE, self::PDF_CERTIFICATE, $this->privateKeyPassword, '', 2, []);
                    $fpdi->Output(basename($filename), $destination);
                    die;
                }
                $this->dompdf->stream($filename, array('Attachment' => 1));
                die;
            } elseif ($destination === self::DEST_STRING) {
                return $this->dompdf->output();
            }

            if (empty($filename)) {
                throw new Exception("Filename is empty");
            }

            // For email attachment
            if ($this->digitallySign()) {
                // Save file with Dompdf
                if (file_put_contents($filename, $this->dompdf->output())) {
                    // Overwrite (Sign) with FPDI & TCPDF
                    $fpdi = $this->fpdi;
                    $pageCount = $fpdi->setSourceFile(Yii::getAlias("@app/$filename"));
                    for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
                        // remove default header/footer
                        $fpdi->SetMargins(0, 0, 0);
                        $fpdi->setPrintHeader(false);
                        $fpdi->setPrintFooter(false);
                        $templateId = $fpdi->importPage($pageNo);
                        $fpdi->AddPage();
                        $fpdi->useTemplate($templateId, 0, 0, null, null, true);
                        // Print Signature Image
                        if (@file_exists(Yii::getAlias(self::PDF_SIGNATURE)) && $pageNo === 1) {
                            $fpdi->Image(Yii::getAlias(self::PDF_SIGNATURE),self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H,'PNG');
                            $fpdi->setSignatureAppearance(self::PDF_SIGNATURE_X,self::PDF_SIGNATURE_Y,self::PDF_SIGNATURE_W,self::PDF_SIGNATURE_H);
                        }
                    }
                    $fpdi->setSignature(self::PDF_CERTIFICATE, self::PDF_CERTIFICATE, $this->privateKeyPassword, '', 2, []);
                    return $fpdi->Output(Yii::getAlias("@app/$filename"), $destination) === '';
                }
            }

            return file_put_contents($filename, $this->dompdf->output());
        }
    }
}