<?php

namespace app\modules\addons\modules\pdf\controllers;

use app\helpers\Liquid;
use app\modules\addons\modules\pdf\models\PdfTemplate;
use app\modules\addons\modules\pdf\models\PdfTemplateSearch;
use app\modules\addons\modules\pdf\services\PdfService;
use Exception;
use Yii;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\web\Response;

/**
 * TemplateController implements the CRUD actions for PdfTemplate model.
 */
class TemplateController extends Controller
{

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\pdf\models\PdfTemplate',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
        ];
    }

    /**
     * Lists all PdfTemplate models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new PdfTemplateSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single PdfTemplate model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new PdfTemplate model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new PdfTemplate();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The PDF Template has been successfully configured.')
            );
            return $this->redirect(['index']);
        }

        return $this->render('create', [
            'model' => $model,
        ]);
    }

    /**
     * Updates an existing PdfTemplate model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The PDF Template has been successfully updated.')
            );
            return $this->redirect(['index']);
        }

        return $this->render('update', [
            'model' => $model,
        ]);
    }

    /**
     * Preview PDF Template.
     * If update is successful, the browser will be redirected to the 'view' page.
     *
     * @return \yii\web\Response
     * @throws Exception
     */
    public function actionPreview()
    {
        $this->layout = false;

        $request = Yii::$app->request;
        $model = new PdfTemplate();

        if ($request->isPost && $model->load($request->post())) {
            $filename = 'document.pdf';
            $password = $model->password;
            $userPassword = Yii::$app->security->generateRandomString(4);

            $html = $model->html;
            $html = Liquid::render($html, []);

            $service = new PdfService();
            $service->setSize($model->paper_size);
            if ($model->paper_size === 'custom') {
                $service->setSize([0, 0, $model->paper_size_width, $model->paper_size_height]);
            }
            $service->setDpi($model->dpi);
            $service->setOrientation($model->orientation);
            $service->setIsJavascriptEnabled((boolean) $model->js_enabled);
            $service->setIsRemoteEnabled((boolean) $model->remote_enabled);
            $service->setHtmlFile($model->html_file);
            $service->setHtml($html);
            $service->setFilename($filename);

            // Set password protection
            if (!empty($password)) {
                $service->setAdminPassword($password);
                $service->setUserPassword($userPassword);
            }

            // Generate PDF as STRING
            if ($data = $service->save(PdfService::DEST_STRING)) {
                Yii::$app->response->format = Response::FORMAT_RAW;
                Yii::$app->response->headers->add('Content-Type', 'text/plain');
                Yii::$app->response->data = base64_encode($data);
                return Yii::$app->response;
            }
        }

        Yii::$app->getSession()->setFlash(
            'danger',
            Yii::t('app', 'An error occurred while processing your request.')
        );

        return $this->goBack();
    }

    /**
     * Deletes an existing PdfTemplate model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();
        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The PDF Template has been successfully deleted.')
        );
        return $this->redirect(['index']);
    }

    /**
     * Finds the PdfTemplate model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return PdfTemplate the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = PdfTemplate::findOne(['id' => $id])) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
    }
}
