<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */

namespace app\modules\addons\modules\pdf\controllers;

use app\helpers\Hashids;
use app\helpers\Liquid;
use app\helpers\SubmissionHelper;
use app\models\FormSubmission;
use app\models\User;
use app\modules\addons\modules\pdf\models\Pdf;
use app\modules\addons\modules\pdf\models\PdfTemplate;
use app\modules\addons\modules\pdf\services\PdfService;
use Yii;
use yii\web\Controller;
use yii\web\ForbiddenHttpException;
use yii\web\NotFoundHttpException;

/**
 * HandleController implements the Public actions for Pdf add-on.
 */
class HandleController extends Controller
{

    /**
     * Download PDF Document
     *
     * @param $id
     * @param $tid
     * @param $sid
     * @throws NotFoundHttpException
     * @throws \yii\base\Exception
     * @throws \yii\base\InvalidConfigException
     */
    public function actionDownload($id, $tid, $sid)
    {
        $model = $this->findModel($id);
        $tid = (int) $tid;

        if ($model && $tid > 0) {
            // Submission Model
            $sid = is_numeric($sid) ? $sid : Hashids::decode($sid);
            $submission = FormSubmission::findOne(['id' => $sid]);
            if (!$submission) {
                throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
            }

            // Login user with api key
            if (!empty($api_key)) {
                $identity = User::findIdentityByAccessToken($api_key);
                Yii::$app->user->login($identity);
            }

            // Checking user access
            if ($model->form->protected_files && !Yii::$app->user->can('viewFormSubmissionFiles', ['model' => $model->form])) {
                throw new ForbiddenHttpException('You are not allowed to access this resource.');
            }

            // Logout user with api key
            if (!empty($api_key)) {
                Yii::$app->user->logout();
            }

            // Form fields
            $fieldsForEmail = $model->form->formData->getFieldsForEmail();
            // Submission Data
            $data = $submission->getSubmissionData();
            // Submission data in an associative array
            $tokens = SubmissionHelper::prepareDataForReplacementToken($data, $fieldsForEmail);
            $fieldData = SubmissionHelper::prepareDataForSubmissionTable($data, $fieldsForEmail);

            // Create multiple PDF files based on PDF Templates
            /** @var PdfTemplate $templates */
            foreach ($model->selectedTemplates as $template) {
                if ($template && $template->id === $tid) { // Validate template
                    $filename = SubmissionHelper::replaceTokens($template->filename, $tokens);
                    $filename = Liquid::render($filename, $tokens);
                    $filename = Yii::getAlias(sprintf('@uploads/%s/%s-%s.pdf',
                        $model->form->getFilesDirectory(),
                        $submission->id,
                        $filename
                    ));
                    $userPassword = Yii::$app->security->generateRandomString(4);

                    // PDF Body
                    $html = SubmissionHelper::replaceTokens($template->html, $tokens);
                    $html = SubmissionHelper::replaceSubmissionTableToken($html, $fieldData);
                    $html = Liquid::render($html, $tokens);

                    $service = new PdfService();
                    $service->setSize($template->paper_size);
                    if ($template->paper_size === 'custom') {
                        $service->setSize([0, 0, $template->paper_size_width, $template->paper_size_height]);
                    }
                    $service->setDpi($template->dpi);
                    $service->setOrientation($template->orientation);
                    $service->setIsJavascriptEnabled((boolean) $template->js_enabled);
                    $service->setIsRemoteEnabled((boolean) $template->remote_enabled);
                    $service->setHtmlFile($template->html_file);
                    $service->setHtml($html);
                    $service->setFilename($filename);
                    $service->setIsSignatureEnabled($model->digital_signature);
                    // Set password protection
                    if (!empty($template->password)) {
                        $service->setAdminPassword($template->password);
                        $service->setUserPassword($userPassword);
                    }
                    $service->save(PdfService::DEST_DOWNLOAD);
                }
            }
        }
    }

    /**
     * Finds the Pdf model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Pdf the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Pdf::findOne(['id' => $id])) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
    }
}