<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */

namespace app\modules\addons\modules\pdf\controllers;

use app\components\User;
use app\helpers\Liquid;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\modules\pdf\models\Pdf;
use app\modules\addons\modules\pdf\models\PdfPdfTemplate;
use app\modules\addons\modules\pdf\models\PdfSearch;
use app\modules\addons\modules\pdf\models\PdfTemplate;
use app\modules\addons\modules\pdf\Module;
use app\modules\addons\modules\pdf\services\PdfService;
use kartik\depdrop\DepDropAction;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\ArrayHelper;
use yii\web\Controller;
use yii\web\NotFoundHttpException;

/**
 * AdminController implements the CRUD actions for Pdf model.
 */
class AdminController extends Controller
{

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'email-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('Pdf')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete', 'download'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = Pdf::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = \app\helpers\ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\pdf\models\Pdf',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                    return [];
                }
            ],
            'email-fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getEmailLabels();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                    return [];
                }
            ],
        ];
    }

    /**
     * Lists all Pdf models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new PdfSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Pdf model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new Pdf model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Pdf();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            // Save templates
            $templates = Yii::$app->request->post('templates');
            if (is_array($templates) && !empty($templates)) {
                foreach ($templates as $template_id) {
                    $item = new PdfPdfTemplate();
                    $item->pdf_template_id = $template_id;
                    $item->pdf_id = $model->id;
                    $item->save();
                }
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The form has been successfully configured.')
            );
            return $this->redirect(['index']);
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = \app\helpers\ArrayHelper::map($forms, 'id', 'name');

        $templates = [];

        $pdfModule = Module::getInstance();
        if ($pdfModule) {
            $templates = $pdfModule->getCurrentUserPdfTemplates()->orderBy('updated_at DESC')->asArray()->all();
            $templates = ArrayHelper::map($templates, 'id', 'name');
        }

        return $this->render('create', [
            'model' => $model,
            'forms' => $forms,
            'templates' => $templates,
        ]);
    }

    /**
     * Updates an existing Pdf model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            // Remove old forms permissions
            PdfPdfTemplate::deleteAll(['pdf_id' => $model->id]);
            // Save templates
            $templates = Yii::$app->request->post('templates');
            if (is_array($templates) && !empty($templates)) {
                foreach ($templates as $template_id) {
                    $item = new PdfPdfTemplate();
                    $item->pdf_template_id = $template_id;
                    $item->pdf_id = $model->id;
                    $item->save();
                }
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The form has been successfully updated.')
            );
            return $this->redirect(['index']);
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = \app\helpers\ArrayHelper::map($forms, 'id', 'name');

        $templates = [];

        $pdfModule = Module::getInstance();
        if ($pdfModule) {
            $templates = $pdfModule->getCurrentUserPdfTemplates()->orderBy('updated_at DESC')->asArray()->all();
            $templates = ArrayHelper::map($templates, 'id', 'name');
        }

        // Get templates of the selected form
        $pdfTemplates = PdfPdfTemplate::find()->select(['pdf_template_id'])->where(['pdf_id' => $model->id])->asArray()->all();
        $pdfTemplates = ArrayHelper::getColumn($pdfTemplates, 'pdf_template_id');

        return $this->render('update', [
            'model' => $model,
            'forms' => $forms,
            'templates' => $templates,
            'pdfTemplates' => $pdfTemplates,
        ]);

    }

    /**
     * Enable / Disable multiple PDF configurations
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {

        $models = Pdf::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing Pdf model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();
        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The form configuration has been successfully deleted.')
        );
        return $this->redirect(['index']);
    }

    /**
     * Download PDF Document
     *
     * @param $id
     * @param $tid
     * @param $submission_id
     * @throws NotFoundHttpException
     * @throws \yii\base\Exception
     * @throws \yii\base\InvalidConfigException
     *
     * @deprecated Instead use HandleController::actionDownload()
     */
    public function actionDownload($id, $tid, $submission_id)
    {
        $model = $this->findModel($id);
        $tid = (int) $tid;

        if ($model && $tid > 0) {
            // Submission Model
            $submission = FormSubmission::findOne(['id' => $submission_id]);
            if (!$submission) {
                throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
            }
            // Form fields
            $fieldsForEmail = $model->form->formData->getFieldsForEmail();
            // Submission Data
            $data = $submission->getSubmissionData();
            // Submission data in an associative array
            $tokens = SubmissionHelper::prepareDataForReplacementToken($data, $fieldsForEmail);
            $fieldData = SubmissionHelper::prepareDataForSubmissionTable($data, $fieldsForEmail);

            // Create multiple PDF files based on PDF Templates
            /** @var PdfTemplate $templates */
            foreach ($model->selectedTemplates as $template) {
                if ($template && $template->id === $tid) { // Validate template
                    $filename = SubmissionHelper::replaceTokens($template->filename, $tokens);
                    $filename = Liquid::render($filename, $tokens);
                    $filename = Yii::getAlias(sprintf('@uploads/%s/%s-%s.pdf',
                        $model->form->getFilesDirectory(),
                        $submission->id,
                        $filename
                    ));
                    $userPassword = Yii::$app->security->generateRandomString(4);

                    // PDF Body
                    $html = SubmissionHelper::replaceTokens($template->html, $tokens);
                    $html = SubmissionHelper::replaceSubmissionTableToken($html, $fieldData);
                    $html = Liquid::render($html, $tokens);

                    $service = new PdfService();
                    $service->setSize($template->paper_size);
                    if ($template->paper_size === 'custom') {
                        $service->setSize([0, 0, $template->paper_size_width, $template->paper_size_height]);
                    }
                    $service->setDpi($template->dpi);
                    $service->setOrientation($template->orientation);
                    $service->setIsJavascriptEnabled((boolean) $template->js_enabled);
                    $service->setIsRemoteEnabled((boolean) $template->remote_enabled);
                    $service->setHtmlFile($template->html_file);
                    $service->setHtml($html);
                    $service->setFilename($filename);
                    $service->setIsSignatureEnabled($model->digital_signature);
                    // Set password protection
                    if (!empty($template->password)) {
                        $service->setAdminPassword($template->password);
                        $service->setUserPassword($userPassword);
                    }
                    $service->save(PdfService::DEST_DOWNLOAD);
                }
            }
        }
    }

    /**
     * Finds the Pdf model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Pdf the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Pdf::findOne(['id' => $id])) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
    }
}
