<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */

namespace app\modules\addons\modules\pdf;

use app\components\rules\RuleEngine;
use app\components\sendinblue\Message as SendinblueMessage;
use app\components\User;
use app\controllers\AjaxController;
use app\helpers\ArrayHelper;
use app\helpers\Liquid;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\pdf\filters\PdfFilters;
use app\modules\addons\modules\pdf\models\Pdf;
use app\modules\addons\modules\pdf\models\PdfPdfTemplate;
use app\modules\addons\modules\pdf\models\PdfTemplate;
use app\modules\addons\modules\pdf\services\PdfService;
use Exception;
use Yii;
use yii\helpers\Url;
use yii\mail\BaseMessage;
use yii\swiftmailer\Message as SwiftMessage;
use yii\web\View;

/**
 * Class Module
 * @package app\modules\addons\modules\pdf
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "pdf";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';
    public static $rendered = false;

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'Pdf' => Pdf::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.mail' => function ($event) {
                $this->onSubmissionMail($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            View::class => [
                'afterRender' => [
                    [self::class, 'onViewAfterRender']
                ],
            ],
            Form::class => [
                'beforeDelete' => [
                    [self::class, 'onFormDeleted']
                ]
            ],
            FormSubmission::class => [
                FormSubmission::EVENT_GET_SUBMISSION_DATA => [
                    [self::class, 'onGetSubmissionData']
                ],
                FormSubmission::EVENT_BEFORE_DELETE => [
                    [self::class, 'onSubmissionDeleted']
                ]
            ],
            SubmissionHelper::class => [
                SubmissionHelper::GET_FIELDS_FOR_FIELD_MAPPING => [
                    [self::class, 'onGetFieldsForFieldMapping']
                ],
            ],
            Liquid::class => [
                Liquid::REGISTER_FILTERS => [
                    [self::class, 'onRegisterFilters']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When Liquid Register Filters is executed
     *
     * @param $event
     */
    public static function onRegisterFilters($event) {
        /** @var Liquid $sender */
        $sender = $event->sender;
        $template = $sender->template;
        $template->registerFilter(new PdfFilters);
        $sender->template = $template;
        $event->sender = $sender;
    }

    /**
     * Event Handler
     * When Get Fields For Field Mapping is executed
     *
     * @param $event
     */
    public static function onGetFieldsForFieldMapping($event)
    {
        /** @var SubmissionHelper $sender */
        $sender = $event->sender;
        $fieldsForFieldMapping = $sender->fieldsForFieldMapping;
        $labels = [
            'pdf_link' => Yii::t('app', 'PDF Link')
        ];
        $sender->fieldsForFieldMapping = array_merge($fieldsForFieldMapping, $labels);
        $event->sender = $sender;
    }

    /**
     * Event Handler
     * When Get Submission Data is executed
     *
     * @param $event
     */
    public static function onGetSubmissionData($event)
    {
        /** @var FormSubmission $sender */
        $sender = $event->sender;
        if ($sender) {
            // Add PDF Links
            $data = $sender->_data;
            $models = Pdf::findAll(['form_id' => $sender->form->id, 'status' => 1]);
            $links = [];
            foreach ($models as $model) {
                foreach ($model->selectedTemplates as $template) {
                    $links[$template->id] = Url::to(['/addons/pdf/handle/download', 'id' => $model->id, 'tid' => $template->id, 'sid' => $sender->hashId], true);
                }
            }
            $sender->_data = array_merge($data, [
                'pdf_link' => $links,
            ]);
        }
        $event->sender = $sender;
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = Pdf::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new Pdf();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();
                $model->refresh();

                foreach ($oModel->templates as $oTemplate) {
                    $template = new PdfPdfTemplate();
                    $template->attributes = $oTemplate->attributes;
                    $template->pdf_id = $model->id;
                    $template->pdf_template_id = $oTemplate->pdf_template_id;
                    $template->isNewRecord = true;
                    $template->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a Form model is deleted
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = Pdf::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * Before a Submission model is deleted
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function onSubmissionDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof FormSubmission && isset($event->sender->id)) {

            $submissionModel = $event->sender;
            $formModel = $submissionModel->form;
            $dataModel = $formModel->formData;
            $fieldsForEmail = $dataModel->getFieldsForEmail();
            $submissionData = $submissionModel->getSubmissionData();
            $tokens = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);

            // Delete PDF Documents
            /** @var Pdf $models */
            $models = Pdf::findAll(['form_id' => $formModel->id, 'status' => 1]);
            foreach ($models as $model) {
                foreach ($model->selectedTemplates as $template) {
                    $filename = SubmissionHelper::replaceTokens($template->filename, $tokens);
                    $filename = Liquid::render($filename, $tokens);
                    $filename = sprintf('%s-%s.pdf',
                        $submissionModel->id,
                        $filename
                    );
                    $filePath = Yii::getAlias(sprintf('@uploads/%s/%s',
                        $formModel->getFilesDirectory(), $filename
                    ));
                    if (file_exists($filePath)) {
                        @unlink($filePath);
                    }
                }
            }
        }
    }

    public function onSubmissionMail($event)
    {
        if (isset($event->type, $event->async, $event->tokens, $event->submission, $event->submission->form, $event->message)) {

            /** @var Form $formModel */
            $formModel = $event->submission->form;
            $dataModel = $formModel->formData;
            $fieldsForEmail = $dataModel->getFieldsForEmail();

            /** @var Pdf $models */
            $models = Pdf::findAll(['form_id' => $formModel->id, 'status' => 1]);

            try {

                foreach ($models as $model) {

                    if (in_array($event->type, $model->notifications)) {

                        $submissionData = $event->submission->getSubmissionData();

                        // Submission data for rule engine
                        $data = SubmissionHelper::prepareDataForRuleEngine($submissionData, $dataModel->getFields());

                        // By default
                        $isValid = true;

                        // Conditional Logic
                        if (!empty($model->conditions)) {
                            $engine = new RuleEngine([
                                'conditions' => $model->conditions,
                                'actions' => [],
                            ]);
                            $isValid = $engine->matches($data);
                        }

                        // If the conditions have been met
                        if ($isValid) {

                            if ($event->message instanceof BaseMessage) {
                                // Create multiple PDF files based on PDF Templates
                                /** @var PdfTemplate $templates */
                                foreach ($model->selectedTemplates as $template) {
                                    if ($template) {
                                        $filename = SubmissionHelper::replaceTokens($template->filename, $event->tokens);
                                        $filename = Liquid::render($filename, $event->tokens);
                                        $filename = sprintf('%s-%s.pdf',
                                            $event->submission->id,
                                            $filename
                                        );
                                        $filePath = Yii::getAlias(sprintf('@uploads/%s/%s',
                                            $formModel->getFilesDirectory(), $filename
                                        ));
                                        $userPassword = Yii::$app->security->generateRandomString(4);

                                        // PDF Body
                                        $html = SubmissionHelper::replaceTokens($template->html, $event->tokens);
                                        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
                                        $html = SubmissionHelper::replaceSubmissionTableToken($html, $fieldData);
                                        $html = Liquid::render($html, $event->tokens);

                                        $service = new PdfService();
                                        $service->setSize($template->paper_size);
                                        if ($template->paper_size === 'custom') {
                                            $service->setSize([0, 0, $template->paper_size_width, $template->paper_size_height]);
                                        }
                                        $service->setDpi($template->dpi);
                                        $service->setOrientation($template->orientation);
                                        $service->setIsJavascriptEnabled((boolean) $template->js_enabled);
                                        $service->setIsRemoteEnabled((boolean) $template->remote_enabled);
                                        $service->setHtmlFile($template->html_file);
                                        $service->setHtml($html);
                                        $service->setFilename($filePath);
                                        $service->setIsSignatureEnabled($model->digital_signature);
                                        // Set password protection
                                        if (!empty($template->password)) {
                                            $service->setAdminPassword($template->password);
                                            $service->setUserPassword($userPassword);
                                        }
                                        if ($service->save(PdfService::DEST_FILE)) {
                                            // Leave available the password by email
                                            if ($event->message instanceof SwiftMessage) {
                                                $swiftMessage = $event->message->getSwiftMessage();
                                                $body = $swiftMessage->getChildren()[0]->getBody();
                                                if (!empty($template->password)) {
                                                    // Replace PDF tokens
                                                    $body = str_replace("{{PDF_USER_PASSWORD}}", $userPassword, $body);
                                                    $body = str_replace("{{PDF_ADMIN_PASSWORD}}", $template->password, $body);
                                                } else {
                                                    // Remove PDF tokens
                                                    $body = str_replace("{{PDF_USER_PASSWORD}}", '', $body);
                                                    $body = str_replace("{{PDF_ADMIN_PASSWORD}}", '', $body);
                                                }
                                                $swiftMessage->getChildren()[0]->setBody($body);
                                            } else if ($event->message instanceof SendinblueMessage) {
                                                $textBody = $event->message->getHtmlBody();
                                                $htmlBody = $event->message->getHtmlBody();
                                                if (!empty($template->password)) {
                                                    // Replace PDF tokens
                                                    $textBody = str_replace("{{PDF_USER_PASSWORD}}", $userPassword, $textBody);
                                                    $textBody = str_replace("{{PDF_ADMIN_PASSWORD}}", $template->password, $textBody);
                                                    $htmlBody = str_replace("{{PDF_USER_PASSWORD}}", $userPassword, $htmlBody);
                                                    $htmlBody = str_replace("{{PDF_ADMIN_PASSWORD}}", $template->password, $htmlBody);
                                                } else {
                                                    // Remove PDF tokens
                                                    $textBody = str_replace("{{PDF_USER_PASSWORD}}", '', $textBody);
                                                    $textBody = str_replace("{{PDF_ADMIN_PASSWORD}}", '', $textBody);
                                                    $htmlBody = str_replace("{{PDF_USER_PASSWORD}}", '', $htmlBody);
                                                    $htmlBody = str_replace("{{PDF_ADMIN_PASSWORD}}", '', $htmlBody);
                                                }
                                                $event->message->setTextBody($textBody);
                                                $event->message->setHtmlBody($htmlBody);
                                            }
                                            // Attach PDF to email message
                                            $event->message->attach($filePath, [
                                                'fileName' => $filename,
                                                'contentType' => 'application/pdf',
                                            ]);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

            } catch (Exception $e) {
                // Hide Error but Log Exception
                Yii::error($e);
            }
        }
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event->sender->context->module->requestedRoute) && $event->sender->context->module->requestedRoute === "form/submissions") {
            $formID = $event->sender->context->request->get("id");
            $formModel = Form::findOne(['id' => $formID]);

            // Flag to improve performance
            if (!self::$rendered && Yii::$app->user->can('configureFormsWithAddons', ['model' => $formModel])) {

                $models = Pdf::findAll(['form_id' => $formModel->id, 'status' => 1]);

                $content = $event->output;

                foreach ($models as $model) {
                    foreach ($model->selectedTemplates as $template) {
                        $content = str_replace(
                            '<i class="far fa-print"></i>',
                            '<i class="far fa-print"></i></a><a class="btn btn-primary btn-pdf" href="#" data-href="' . Url::to(['/addons/pdf/handle/download', 'id' => $model->id, 'tid' => $template->id], true) . '" title="'.Yii::t('app', 'Download PDF') . ' - ' . $model->name . ' - '. $template->name .'"> <i class="far fa-download"></i> ',
                            $content);
                    }
                }

                $content .= <<<EOT
<script>
window.addEventListener('load', function(){
  $('body').on('click', '.btn-pdf', function(event) {
        event.preventDefault();
        var hash = location.hash.substr(1),
        submission_id = hash.replace(/[^0-9]/g, ''),
        redirectWindow = window.open($(this).data('href') + '&sid=' + submission_id, '_blank');
        redirectWindow.location;
  });
});
</script>
EOT;

                $event->output =  $content;

                self::$rendered = true;
            }
        }
    }


    /**
     * Return ActiveQuery of PDFTemplate models accessible by this user
     *
     * Important! Don't select columns with the returned active query,
     * because we can't get unexpected behaviors
     *
     * @return \yii\db\ActiveQuery
     */
    public function getCurrentUserPdfTemplates()
    {
        /** @var User $currentUser */
        $currentUser = Yii::$app->user;

        // If user can view all forms
        if ($currentUser->can("viewForms")) {
            // Return all conditional email templates
            return PdfTemplate::find();
        }

        // Forms
        $formModels = $currentUser->forms()->asArray()->all();
        $formIds = ArrayHelper::getColumn($formModels, 'id');

        // PDFs
        $pdfModels = Pdf::find()->where(['in', 'form_id', $formIds])->asArray()->all();
        $pdfIds = ArrayHelper::getColumn($pdfModels, 'id');

        // Shared PDF Templates
        $templateModels = PdfPdfTemplate::find()->where(['in', 'pdf_id', $pdfIds])->asArray()->all();
        $templateIds = ArrayHelper::getColumn($templateModels, 'pdf_template_id');

        // Own PDF templates
        $templates = PdfTemplate::find()->where(['created_by' => $currentUser->id]);

        return $templates->union(PdfTemplate::find()->where(['in', 'id', $templateIds]));
    }

}