<?php

namespace app\modules\addons\modules\paypal_invoicing\services;

use Exception;
use yii\helpers\Json;
use app\helpers\ArrayHelper;

class PaypalInvoicingService
{
    /** @var string Base Url */
    protected $baseUrl = "https://api.sandbox.paypal.com";

    /** @var string Live Base Url */
    protected $liveBaseUrl = "https://api.paypal.com";

    /** @string Mode */
    protected $mode = "sandbox";

    /** @string Client ID */
    protected $clientId;

    /** @string Client Secret */
    protected $clientSecret;

    /** @string Access Token */
    protected $accessToken;

    /** @string Access Token */
    protected $expiresIn;

    /** @string Access Token */
    protected $tokenType;

    /** @string App Id */
    protected $appId;

    /** @string Nonce */
    protected $nonce;

    /** @string Access Token */
    protected $scope;

    /** @string Authorization */
    protected $authorization;

    protected $api;

    public function __construct($mode, $clientId, $clientSecret)
    {
        if (empty($mode)) {
            throw new Exception('The mode is empty');
        }
        if (empty($clientId)) {
            throw new Exception('The client id is empty');
        }
        if (empty($clientSecret)) {
            throw new Exception('The client secret is empty');
        }
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        if ($mode === 'live') {
            $this->mode = $mode;
            $this->baseUrl = $this->liveBaseUrl;
        }
        $this->api = new PaypalInvoicingRestApi();
        $this->api->setBaseUrl($this->baseUrl);
    }

    /**
     * Refresh OAuth
     *
     * @return bool
     */
    protected function refreshAuthorization()
    {

        $result = $this->api->getCredentials($this->baseUrl, $this->clientId, $this->clientSecret);

        if (!empty($result)) {
            $auth = is_string($result) ? Json::decode($result) : $result;
            if (isset($auth['access_token'], $auth['expires_in'], $auth['token_type'], $auth['scope'])) {
                $this->accessToken = $auth['access_token'];
                $this->expiresIn = $auth['expires_in'];
                $this->tokenType = $auth['token_type'];
                $this->appId = $auth['app_id'];
                $this->nonce = $auth['nonce'];
                $this->scope = $auth['scope'];
                $auth['unix_timestamp'] = time();
                $this->authorization = Json::encode($auth);
                return true;
            }
        }

        return false;
    }

    /**
     * @return string Authorization
     */
    public function getAuthorization()
    {
        $this->refreshAuthorization();
        return $this->authorization;
    }

    /**
     * @param $accessToken
     */
    public function setAccessToken($accessToken)
    {
        $this->accessToken = $accessToken;
    }

    public function getTemplates()
    {
        $this->api->setAuthorization($this->accessToken);
        $result = $this->api->get("/v2/invoicing/templates?fields=none");

        $templates = [];

        if (!empty($result)) {
            $templateCollection  = is_string($result) ? Json::decode($result) : $result;
            if (is_array($templateCollection) && !empty($templateCollection['templates'])) {
                foreach($templateCollection['templates'] as $template) {
                    $template = (array) $template;
                    if (isset($template['id'], $template['name'])) {
                        $templates[$template['id']] = $template['name'];
                    }
                }
            }
        }

        return $templates;
    }

    public function getTemplate($template_id)
    {
        $this->api->setAuthorization($this->accessToken);
        $result = $this->api->get("/v2/invoicing/templates/{$template_id}");

        $template = [];

        if (!empty($result)) {
            $template  = is_string($result) ? Json::decode($result) : $result;
        }

        return $template;
    }

    public function generateInvoiceNumber()
    {
        $this->api->setAuthorization($this->accessToken);
        $result = $this->api->post("/v2/invoicing/generate-next-invoice-number");

        $number = false;

        if (!empty($result)) {
            $result  = is_string($result) ? Json::decode($result) : $result;
            $number = isset($result['invoice_number']) ? $result['invoice_number'] : false;
        }

        return $number;
    }

    public function createDraftInvoice($invoice)
    {
        $invoice = ArrayHelper::removeEmptyElements($invoice);
        $this->api->setAuthorization($this->accessToken);
        $response = $this->api->post("/v2/invoicing/invoices", $invoice);

        if (!empty($response)) {
            $response  = is_string($response) ? Json::decode($response) : $response;
            if (!empty($response['href'])) {
                $response = $this->api->get($response['href']);
                $response  = is_string($response) ? Json::decode($response) : $response;
            } else {
                $response = false;
            }
        }
        return $response;
    }

    public function sendInvoice($invoice_id, $settings)
    {
        $this->api->setAuthorization($this->accessToken);
        return $this->api->post("/v2/invoicing/invoices/{$invoice_id}/send", $settings);
    }

}