<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal_invoicing\helpers;

use Yii;
use PayPal\Api\Payment;
use PayPal\Rest\ApiContext;
use PayPal\Api\PaymentExecution;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Exception\PayPalConnectionException;

class PayPalService
{

    /**
     * Use an ApiContext object to authenticate API calls
     * @var ApiContext
     */
    public $apiContext;

    /**
     * PayPalService constructor.
     */
    public function __construct($clientId, $clientSecret, $mode)
    {
        // Replace these values by entering your own ClientId and Secret
        // by visiting https://developer.paypal_invoicing.com/webapps/developer/applications/myapps
        /** @var \PaypalInvoicing\Rest\ApiContext $apiContext */
        $this->apiContext = $this->getApiContext($clientId, $clientSecret, $mode);
    }

    /**
     * Get ApiContext object to authenticate API calls.
     *
     * @param $clientId
     * @param $clientSecret
     * @param string $mode
     * @return ApiContext
     */
    public function getApiContext($clientId, $clientSecret, $mode = "sandbox")
    {
        // ### Api context
        // Use an ApiContext object to authenticate
        // API calls. The clientId and clientSecret for the
        // OAuthTokenCredential class can be retrieved from
        // developer.paypal_invoicing.com
        $apiContext = new ApiContext(
            new OAuthTokenCredential(
                $clientId,
                $clientSecret
            )
        );

        // Comment this line out and uncomment the PP_CONFIG_PATH
        // 'define' block if you want to use static file
        // based configuration
        $apiContext->setConfig(
            array(
                'mode' => $mode,
                // 'log.LogEnabled' => true,
                // 'log.FileName' => '../PayPal.log',
                // 'log.LogLevel' => 'DEBUG', // PLEASE USE `INFO` LEVEL FOR LOGGING IN LIVE ENVIRONMENTS
                // 'cache.enabled' => true,
                // 'http.CURLOPT_CONNECTTIMEOUT' => 30
                // 'http.headers.PayPal-Partner-Attribution-Id' => '123123123'
                //'log.AdapterFactory' => '\PayPal\Log\DefaultLogFactory' // Factory class implementing \PayPal\Log\PayPalLogFactory
            )
        );

        // Partner Attribution Id
        // Use this header if you are a PayPal partner. Specify a unique BN Code to receive revenue attribution.
        // To learn more or to request a BN Code, contact your Partner Manager or visit the PayPal Partner Portal
        // $apiContext->addRequestHeader('PayPal-Partner-Attribution-Id', '123123123');
        return $apiContext;
    }

    /**
     * Execute an payment
     *
     * @param $paymentId
     * @param $payerID
     * @return Payment|boolean
     */
    public function executePayment($paymentId, $payerID)
    {
        try {
            // Get the payment Object by passing paymentId
            // payment id was previously stored in session in
            $payment = Payment::get($paymentId, $this->apiContext);
            // ### Payment Execute
            // PaymentExecution object includes information necessary
            // to execute a PayPal account payment.
            // The payer_id is added to the request query parameters
            // when the user is redirected from paypal_invoicing back to your site
            $execution = new PaymentExecution();
            $execution->setPayerId($payerID);
            // Execute the payment
            return $payment->execute($execution, $this->apiContext);
        } catch (PayPalConnectionException $pce) {
            // Don't spit out errors or use "exit" like this in production code
            // echo '<pre>';var_dump(json_decode($pce->getData()), $options);exit;
            if (YII_DEBUG) {
                die(var_dump([
                    'errors' => true,
                    'details' => json_decode($pce->getData()),
                ]));
            }
        }

        return false;
    }
}