<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal_invoicing\controllers;

use app\components\User;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingRecipient;
use Yii;
use yii\filters\AccessControl;
use yii\helpers\Json;
use yii\web\Controller;
use yii\filters\VerbFilter;
use yii\web\NotFoundHttpException;
use app\models\Form;
use app\helpers\ArrayHelper;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicing;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingItem;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingSearch;
use app\modules\addons\modules\paypal_invoicing\services\PaypalInvoicingService;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingInvoiceSearch;

/**
 * AdminController implements the CRUD actions for PaypalInvoicing model.
 */
class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('PaypalInvoicing')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'invoices', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = PaypalInvoicing::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicing',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all PaypalInvoicing models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new PaypalInvoicingSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionInvoices($id)
    {
        $searchModel = new PaypalInvoicingInvoiceSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('invoices', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * @return string|\yii\web\Response
     * @throws \Exception
     */
    public function actionCreate()
    {
        $templates = [];
        $connected = false;

        $model = new PaypalInvoicing();
        $recipientModel = new PaypalInvoicingRecipient();
        $request = Yii::$app->request;

        if ($request->isPost && $model->load($request->post())) {

            if ($model->mode === 'sandbox') {
                $service = new PaypalInvoicingService($model->mode, $model->sandbox_client_id, $model->sandbox_client_secret);
                $model->sandbox_oauth = $service->getAuthorization();
                $templates = $service->getTemplates();
            } else {
                $service = new PaypalInvoicingService($model->mode, $model->client_id, $model->client_secret);
                $model->oauth = $service->getAuthorization();
                $templates = $service->getTemplates();
            }

            $connected = true;

            if ($model->template_id && $model->validate()) {
                $model->items = Yii::$app->request->post('PaypalInvoicingItem', []);
                $model->recipients = Yii::$app->request->post('PaypalInvoicingRecipient', []);
                $model->save(false);
                Yii::$app->getSession()->setFlash(
                    'success',
                    Yii::t('app', 'Your form has been successfully configured.')
                );
                return $this->redirect(['index']);
            }
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'itemModel' => new PaypalInvoicingItem,
            'recipientModel' => $recipientModel,
            'forms' => $forms,
            'templates' => $templates,
            'connected' => $connected,
        ]);
    }

    /**
     * @param $id
     * @return string|\yii\web\Response
     * @throws \Exception
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $connected = false;
        $templates = [];

        $request = Yii::$app->request;

        if ($model->load($request->post()) && $model->validate()) {
            $items = Yii::$app->request->post('PaypalInvoicingItem', []);
            $recipients = Yii::$app->request->post('PaypalInvoicingRecipient', []);
            if (empty($items) || empty($recipients)) {
                if ($model->mode === 'sandbox') {
                    $service = new PaypalInvoicingService($model->mode, $model->sandbox_client_id, $model->sandbox_client_secret);
                    $model->sandbox_oauth = $service->getAuthorization();
                } else {
                    $service = new PaypalInvoicingService($model->mode, $model->client_id, $model->client_secret);
                    $model->oauth = $service->getAuthorization();
                }
            } else {
                $model->items = $items;
                $model->recipients = $recipients;
            }
            $model->save(false);
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'Your form has been successfully configured.')
            );
            return $this->redirect(['index']);

        }

        try {

            if ($model->mode === 'sandbox') {
                $service = new PaypalInvoicingService($model->mode, $model->sandbox_client_id, $model->sandbox_client_secret);
                $oauth = Json::decode($model->sandbox_oauth);
            } else {
                $service = new PaypalInvoicingService($model->mode, $model->client_id, $model->client_secret);
                $oauth = Json::decode($model->oauth);
            }

            if (isset($oauth['access_token'])) {
                $service->setAccessToken($oauth['access_token']);
                $templates = $service->getTemplates();
                if (!empty($templates)) {
                    $connected = true;
                }
            }

        } catch (\Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'itemModel' => new PaypalInvoicingItem,
            'recipientModel' => new PaypalInvoicingRecipient,
            'forms' => $forms,
            'templates' => $templates,
            'connected' => $connected,
        ]);
    }

    /**
     * Enable / Disable multiple Payment Forms
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = PaypalInvoicing::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * @param $id
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();
        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The payment form configuration has been successfully deleted.')
        );
        return $this->redirect(['index']);
    }

    /**
     * Finds the PaypalInvoicing model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return PaypalInvoicing the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = PaypalInvoicing::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}