<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal_invoicing;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicing;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingInvoice;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingItem;
use app\modules\addons\modules\paypal_invoicing\models\PaypalInvoicingRecipient;
use app\modules\addons\modules\paypal_invoicing\services\PaypalInvoicingService;
use Exception;
use Yii;
use yii\helpers\Json;

/**
 * Class Module
 * @package app\modules\addons\modules\paypal_invoicing
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "paypal_invoicing";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'PaypalInvoicing' => PaypalInvoicing::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = PaypalInvoicing::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new PaypalInvoicing();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new PaypalInvoicingItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->paypal_invoicing_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }

                foreach ($oModel->recipients as $oRecipient) {
                    $recipient = new PaypalInvoicingRecipient();
                    $recipient->attributes = $oRecipient->attributes;
                    $recipient->id = null;
                    $recipient->paypal_invoicing_id = $model->id;
                    $recipient->form_id = $event->form->id;
                    $recipient->isNewRecord = true;
                    $recipient->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = PaypalInvoicing::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {

        $result = false;

        $models = PaypalInvoicing::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var \app\models\FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                try {

                    /** @var PaypalInvoicingService $service */
                    $service = null;

                    // Set or Refresh access token if it has expired
                    if ($model->mode === 'live') {
                        $service = new PaypalInvoicingService($model->mode, $model->client_id, $model->client_secret);
                        $oauth = Json::decode($model->oauth);
                        if (isset($oauth['access_token'], $oauth['expires_in'], $oauth['unix_timestamp'])) {
                            if (time() < ((int) $oauth['unix_timestamp'] + (int) $oauth['expires_in'])) {
                                $model->oauth = $service->getAuthorization();
                                $model->save();
                            } else {
                                $service->setAccessToken($oauth['access_token']);
                            }
                        }
                    } else {
                        $service = new PaypalInvoicingService($model->mode, $model->sandbox_client_id, $model->sandbox_client_secret);
                        $oauth = Json::decode($model->sandbox_oauth);
                        if (isset($oauth['access_token'], $oauth['expires_in'], $oauth['unix_timestamp'])) {
                            if (time() > ((int) $oauth['unix_timestamp'] + (int) $oauth['expires_in'])) {
                                $model->sandbox_oauth = $service->getAuthorization();
                                $model->save();
                            } else {
                                $service->setAccessToken($oauth['access_token']);
                            }
                        }
                    }

                    if (!empty($model->template_id) && $service instanceof PaypalInvoicingService) {

                        /**
                         * 01. Generate invoice number
                         */
                        $invoiceNumber = $service->generateInvoiceNumber();

                        /**
                         * 02. Get Invoice Template
                         */
                        $template = $service->getTemplate($model->template_id);

                        /**
                         * 03. Create draft invoice
                         */
                        // Details
                        $detail = [
                            'invoice_number' => $invoiceNumber, // Maximum length: 127.
                            'invoice_date' => date('Y-m-d'),
                        ];
                        if (!empty($template['template_info']['detail'])) {
                            $detail = array_merge($template['template_info']['detail'], $detail);
                        }
                        // Invoicer
                        $invoicer = [];
                        if (!empty($template['template_info']['invoicer'])) {
                            $invoicer = array_merge($template['template_info']['invoicer'], $invoicer);
                        }
                        // Primary recipients
                        $primary_recipients = [];
                        foreach ($model->recipients as $r) {
                            if (!empty($r->billing_email_address)) {
                                $email = !empty($submissionData[$r->billing_email_address]) ? $submissionData[$r->billing_email_address] : '';
                                $recipient = [
                                    'billing_info' => [
                                        'email_address' => $email,
                                        'name' => [
                                            'given_name' => !empty($submissionData[$r->billing_name]) ? substr($submissionData[$r->billing_name], 0, 140) : '',
                                            'surname' => !empty($submissionData[$r->billing_surname]) ? substr($submissionData[$r->billing_surname], 0, 140) : '',
                                        ],
                                        'address' => [
                                            'address_line_1' => !empty($submissionData[$r->billing_address_line_1]) ? substr($submissionData[$r->billing_address_line_1], 0, 300) : '',
                                            'admin_area_2' => !empty($submissionData[$r->billing_admin_area_2]) ? substr($submissionData[$r->billing_admin_area_2], 0, 120) : '',
                                            'admin_area_1' => !empty($submissionData[$r->billing_admin_area_1]) ? substr($submissionData[$r->billing_admin_area_1], 0, 300) : '',
                                            'postal_code' => !empty($submissionData[$r->billing_postal_code]) ? substr($submissionData[$r->billing_postal_code], 0, 60) : '',
                                            'country_code' => !empty($submissionData[$r->billing_country_code]) ? substr($submissionData[$r->billing_country_code], 0, 2) : '',
                                        ],
                                        'additional_info' => !empty($submissionData[$r->billing_additional_value]) ? substr($submissionData[$r->billing_additional_value], 0, 40) : '',
                                    ],
                                    'shipping_info' => [
                                        'name' => [
                                            'given_name' => !empty($submissionData[$r->shipping_name]) ? substr($submissionData[$r->shipping_name], 0, 140) : '',
                                            'surname' => !empty($submissionData[$r->shipping_surname]) ? substr($submissionData[$r->shipping_surname], 0, 140) : '',
                                        ],
                                        'address' => [
                                            'address_line_1' => !empty($submissionData[$r->shipping_address_line_1]) ? substr($submissionData[$r->shipping_address_line_1], 0, 300) : '',
                                            'admin_area_2' => !empty($submissionData[$r->shipping_admin_area_2]) ? substr($submissionData[$r->shipping_admin_area_2], 0, 120) : '',
                                            'admin_area_1' => !empty($submissionData[$r->shipping_admin_area_1]) ? substr($submissionData[$r->shipping_admin_area_1], 0, 300) : '',
                                            'postal_code' => !empty($submissionData[$r->shipping_postal_code]) ? substr($submissionData[$r->shipping_postal_code], 0, 60) : '',
                                            'country_code' => !empty($submissionData[$r->shipping_country_code]) ? substr($submissionData[$r->shipping_country_code], 0, 2) : '',
                                        ]
                                    ],
                                    'language' => !empty($submissionData[$r->billing_language]) ? substr($submissionData[$r->billing_language], 0, 10) : '',
                                ];

                                // Phone Numbers
                                if (!empty($r->billing_phone_country_code) && !empty($r->billing_phone_national_number)) {
                                    // Required fields
                                    if (!empty($submissionData[$r->billing_phone_country_code]) && !empty($submissionData[$r->billing_phone_national_number])) {
                                        $recipient['billing_info']['phones'] = [
                                            [
                                                'country_code' => !empty($submissionData[$r->billing_phone_country_code]) ? substr($submissionData[$r->billing_phone_country_code], 0, 3) : '',
                                                'national_number' => !empty($submissionData[$r->billing_phone_national_number]) ? substr($submissionData[$r->billing_phone_national_number], 0, 14) : '',
                                                'phone_type' => !empty($submissionData[$r->billing_phone_type]) ? substr($submissionData[$r->billing_phone_type], 0, 6) : 'OTHER'
                                            ]
                                        ];
                                    }
                                }

                                array_push($primary_recipients, $recipient);
                            }
                        }
                        if (!empty($template['template_info']['primary_recipients'])) {
                            $primary_recipients = array_push($template['template_info']['primary_recipients'], $primary_recipients);
                        }
                        // Additional recipients
                        $additional_recipients = [];
                        if (!empty($template['template_info']['additional_recipients'])) {
                            $additional_recipients = array_push($template['template_info']['additional_recipients'], $additional_recipients);
                        }
                        // Items
                        $items = [];
                        foreach ($model->items as $i) {
                            if (!empty($i->name) && !empty($i->price)) {
                                $price = !empty($submissionData[$i->price]) ? $submissionData[$i->price] : 0;
                                $price = is_array($price) ? array_sum($price) : $price;
                                if ($price === 0) {
                                    continue;
                                }
                                $quantity = !empty($i->quantity) && !empty($submissionData[$i->quantity]) ? $submissionData[$i->quantity] : 1;
                                $quantity = is_array($quantity) ? array_sum($quantity) : $quantity;
                                $item = [
                                    'name' => $i->name,
                                    'unit_amount' => [
                                        'currency_code' => !empty($detail['currency_code']) ? $detail['currency_code'] : 'USD',
                                        'value' => $price,
                                    ],
                                    'quantity' => $quantity,
                                    'unit_of_measure' => !empty($template['unit_of_measure']) ? $template['unit_of_measure'] : 'QUANTITY',
                                ];
                                if (!empty($i->description)) {
                                    $description = !empty($submissionData[$i->description]) ? $submissionData[$i->description] : '';
                                    $description = is_array($description) ? implode(', ', $description) : $description;
                                    $item['description'] = substr($description, 0, 1000);
                                }
                                if (!empty($i->tax_name) && !empty($i->tax_percent)) {
                                    $item['tax'] = [
                                        'name' => $i->tax_name,
                                        'percent' => $i->tax_percent,
                                    ];
                                }
                                if (!empty($i->discount_percent)) {
                                    $item['discount'] = [
                                        'percent' => $i->discount_percent
                                    ];
                                }
                                array_push($items, $item);
                            }
                        }
                        if (!empty($template['template_info']['items'])) {
                            $items = array_push($template['template_info']['items'], $items);
                        }
                        // Configuration
                        $configuration = [];
                        if (!empty($template['template_info']['configuration'])) {
                            $configuration = array_merge($template['template_info']['configuration'], $configuration);
                        }
                        // Amount
                        $amount = [];
                        if (!empty($template['template_info']['amount'])) {
                            $amount = array_merge($template['template_info']['amount'], $amount);
                        }

                        $draft = [
                            'detail' => $detail,
                            'invoicer' => $invoicer,
                            'primary_recipients' => $primary_recipients,
                            'additional_recipients' => $additional_recipients,
                            'items' => $items,
                            'configuration' => $configuration,
                            'amount' => $amount,
                        ];

                        $invoice = $service->createDraftInvoice($draft);

                        /**
                         * 04. Send Invoice
                         */
                        $success = $invoice && !empty($invoice['id']);
                        if ($success) {
                            $service->sendInvoice($invoice['id'], [
                                'send_to_invoicer' => true,
                                'send_to_recipient' => true,
                            ]);
                        }

                        /**
                         * 05. Record response
                         */
                        $invoiceModel = new PaypalInvoicingInvoice();
                        $invoiceModel->form_id = $model->form_id;
                        $invoiceModel->paypal_invoicing_id = $model->id;
                        $invoiceModel->submission_id = $submissionModel->id;
                        $invoiceModel->data = Json::encode($invoice);
                        $invoiceModel->success = $success ? 1 : 0;
                        $invoiceModel->save();

                    }

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;
    }
}