<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal\services;

use Yii;
use yii\helpers\Url;

/**
 * Class PaypalService
 * @package app\modules\addons\modules\paypal\services
 */
class PaypalService
{

    /**
     * Use an ApiContext object to authenticate API calls
     * @var \PayPal\Rest\ApiContext
     */
    public $apiContext;

    /**
     * PaypalService constructor.
     * @param $clientId
     * @param $clientSecret
     * @param $mode
     */
    public function __construct($clientId, $clientSecret, $mode)
    {

        require_once(Yii::getAlias('@addons/paypal/services/PayPal-PHP-SDK/autoload.php'));

        // Replace these values by entering your own ClientId and Secret
        // by visiting https://developer.paypal.com/webapps/developer/applications/myapps
        /** @var \Paypal\Rest\ApiContext $apiContext */
        $this->apiContext = $this->getApiContext($clientId, $clientSecret, $mode);
    }

    /**
     * Get ApiContext object to authenticate API calls.
     *
     * @param $clientId
     * @param $clientSecret
     * @param string $mode
     * @return \PayPal\Rest\ApiContext
     */
    public function getApiContext($clientId, $clientSecret, $mode = "sandbox")
    {
        // ### Api context
        // Use an ApiContext object to authenticate
        // API calls. The clientId and clientSecret for the
        // OAuthTokenCredential class can be retrieved from
        // developer.paypal.com
        $apiContext = new \PayPal\Rest\ApiContext(
            new \PayPal\Auth\OAuthTokenCredential(
                $clientId,
                $clientSecret
            )
        );

        // Comment this line out and uncomment the PP_CONFIG_PATH
        // 'define' block if you want to use static file
        // based configuration
        $apiContext->setConfig(
            array(
                'mode' => $mode,
                // 'log.LogEnabled' => true,
                // 'log.FileName' => '../PayPal.log',
                // 'log.LogLevel' => 'DEBUG', // PLEASE USE `INFO` LEVEL FOR LOGGING IN LIVE ENVIRONMENTS
                // 'cache.enabled' => true,
                // 'http.CURLOPT_CONNECTTIMEOUT' => 30
                // 'http.headers.PayPal-Partner-Attribution-Id' => '123123123'
                //'log.AdapterFactory' => '\PayPal\Log\DefaultLogFactory' // Factory class implementing \PayPal\Log\PayPalLogFactory
            )
        );

        // Partner Attribution Id
        // Use this header if you are a PayPal partner. Specify a unique BN Code to receive revenue attribution.
        // To learn more or to request a BN Code, contact your Partner Manager or visit the PayPal Partner Portal
        // $apiContext->addRequestHeader('PayPal-Partner-Attribution-Id', '123123123');
        return $apiContext;
    }

    /**
     * Create payment
     *
     * @param $model
     * @param $formModel
     * @param $submissionModel
     * @param $submissionData
     * @return string|null
     */
    public function createPayment($model, $formModel, $submissionModel, $submissionData)
    {

        // ### Payer
        $payer = new \PayPal\Api\Payer();
        $payer->setPaymentMethod($model->payment_method);

        // ### Itemized information
        $items = [];
        $subtotal = 0;
        $shipping = isset($model->shipping, $submissionData[$model->shipping]) &&
        !empty($submissionData[$model->shipping]) ? $submissionData[$model->shipping] : 0;
        $handling = isset($model->handling, $submissionData[$model->handling]) &&
        !empty($submissionData[$model->handling]) ? $submissionData[$model->handling] : 0;
        $tax = isset($model->tax, $submissionData[$model->tax]) &&
        !empty($submissionData[$model->tax]) ? $submissionData[$model->tax] : 0;

        // Sum values if field value is an array
        $shipping = is_array($shipping) ? array_sum($shipping) : $shipping;
        $handling = is_array($handling) ? array_sum($handling) : $handling;
        $tax = is_array($tax) ? array_sum($tax) : $tax;

        // Currency
        $currency = $model->currency;
        if (!empty($currency) && !empty($submissionData[$currency])) {
            $currency = is_array($submissionData[$currency]) ? implode('', $submissionData[$currency]) : $submissionData[$currency];
        }

        foreach ($model->items as $itemModel) {
            // Check field value
            $quantity = isset($itemModel->quantity, $submissionData[$itemModel->quantity]) ?
                $submissionData[$itemModel->quantity] : 1;

            $price = isset($itemModel->price, $submissionData[$itemModel->price]) ?
                $submissionData[$itemModel->price] : 0;

            // Sum values if field value is an array
            $quantity = is_array($quantity) ? array_sum($quantity) : $quantity;
            $price = is_array($price) ? array_sum($price) : $price;

            // ### Itemized information
            // (Optional) Lets you specify item wise
            // information
            if (($quantity * $price) > 0) {
                $item = new \PayPal\Api\Item();
                $item->setName($itemModel->description)
                    ->setCurrency($currency)
                    ->setQuantity($quantity)
                    ->setPrice($price);
                array_push($items, $item);
                $subtotal += $quantity * $price;
            }
        }

        $itemList = new \PayPal\Api\ItemList();
        $itemList->setItems($items);

        // ### Additional payment details
        $details = new \PayPal\Api\Details();
        $details->setShipping($shipping)
            ->setTax($tax)
            ->setHandlingFee($handling)
            ->setSubtotal($subtotal);

        // ### Amount
        $total = $subtotal + $shipping + $tax + $handling;
        $amount = new \PayPal\Api\Amount();
        $amount->setCurrency($currency)
            ->setTotal($total)
            ->setDetails($details);

        // ### Transaction
        $transaction = new \PayPal\Api\Transaction();
        $transaction->setAmount($amount)
            ->setItemList($itemList)
            ->setDescription($model->payment_description)
            ->setInvoiceNumber(uniqid());

        // ### Redirect urls
        $redirectUrls = new \PayPal\Api\RedirectUrls();
        $redirectUrls->setReturnUrl(Url::to(['/addons/paypal/check/payment', 'id' => $model->id, 'form_id' => $formModel->id, 'submission_id' => $submissionModel->id], true))
            ->setCancelUrl($model->cancel_url);

        // ### Payment
        $payment = new \PayPal\Api\Payment();
        $payment->setIntent($model->intent)
            ->setPayer($payer)
            ->setRedirectUrls($redirectUrls)
            ->setTransactions([$transaction]);

        // ### Create Payment
        $payment->create($this->apiContext);

        // ### Get redirect url
        return $payment->getApprovalLink();
    }

    /**
     * Execute an payment
     *
     * @param $paymentId
     * @param $payerID
     * @return \PayPal\Api\Payment|boolean
     */
    public function executePayment($paymentId, $payerID)
    {
        try {
            // Get the payment Object by passing paymentId
            // payment id was previously stored in session in
            $payment = \PayPal\Api\Payment::get($paymentId, $this->apiContext);
            // ### Payment Execute
            // \PayPal\Api\PaymentExecution object includes information necessary
            // to execute a PayPal account payment.
            // The payer_id is added to the request query parameters
            // when the user is redirected from paypal back to your site
            $execution = new \PayPal\Api\PaymentExecution();
            $execution->setPayerId($payerID);
            // Execute the payment
            return $payment->execute($execution, $this->apiContext);
        } catch (\PayPal\Exception\PayPalConnectionException $pce) {
            // Log error
            Yii::error($pce);
        }

        return false;
    }
}