<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal\controllers;

use app\components\rules\RuleEngine;
use app\helpers\EventHelper;
use app\helpers\Hashids;
use app\helpers\SubmissionHelper;
use app\models\FormSubmission;
use Exception;
use Yii;
use yii\helpers\Url;
use yii\web\NotFoundHttpException;
use yii\web\Response;
use yii\web\Controller;
use app\modules\addons\modules\paypal\models\Paypal;
use app\modules\addons\modules\paypal\models\PaypalPayment;
use app\modules\addons\modules\paypal\services\PaypalService;

/**
 * CheckController implements the CRUD actions for Paypal model.
 */
class CheckController extends Controller
{

    public $payment = null;

    /**
     * Redirect Users to PayPal Approval Link
     *
     * This endpoint will generate the payment object in PayPal and redirect the user to the PayPal Gateway.
     * After the payment, the user will be redirected to the "Return URL",
     * and all the system events will be triggered, including webhooks and conditional emails.
     *
     * This method adds the following GET endpoint:
     * - https://example.com/addons/paypal/check/approval?sid={HashID}
     * - https://example.com/addons/paypal/check/approval?sid={SubmissionID}
     * Where "sid" can be or the Submission Hash ID or the Submission ID.
     *
     * @param int $sid Submission ID
     * @return Response
     * @throws NotFoundHttpException
     */
    public function actionApproval($sid)
    {
        $sid = is_numeric($sid) ? $sid : Hashids::decode($sid);
        /** @var FormSubmission $submissionModel */
        $submissionModel = FormSubmission::findOne(['id' => $sid]);

        if ($submissionModel === null) {
            throw new NotFoundHttpException(Yii::t('app', 'The requested page does not exist.'));
        }

        // Verify if this submission has a previous payment
        $previousPayment = PaypalPayment::findOne(['submission_id' => $submissionModel->id]);
        if ($previousPayment) {
            // Redirect to Success Url
            return $this->redirect($previousPayment->paypal->return_url);
        }

        $formModel = $submissionModel->form;
        $dataModel = $formModel->formData;
        $event = FormSubmission::STATUS_ACCEPTED;
        $models = Paypal::findAll(['form_id' => $formModel->id, 'status' => 1]);
        $submissionData = $submissionModel->getSubmissionData();
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {
                try {
                    // Create PayPal Payment and redirect to Approval Link
                    $service = new PaypalService($model->client_id, $model->client_secret, $model->mode);
                    $approvalLink = $service->createPayment($model, $formModel, $submissionModel, $submissionData);
                    return $this->redirect($approvalLink);
                } catch (Exception $e) {
                    // Log exception
                    Yii::error($e);
                }
            }
        }

        // Redirect to Form Page
        return $this->redirect(Url::to([
            '/app/forms',
            'slug' => $formModel->slug
        ], true));
    }

    /**
     * Execute PayPal Payment
     *
     * @param int $id Paypal Model ID
     * @param int $form_id Form Model ID
     * @param int $submission_id Submission Model ID
     * @return Response
     */
    public function actionPayment($id, $form_id, $submission_id)
    {
        $model = Paypal::findOne(['id' => $id]);

        if ($model !== null && isset($_GET['paymentId'], $_GET['PayerID'])) {

            // Execute payment
            $payPalService = new PaypalService(
                $model->client_id, $model->client_secret, $model->mode
            );
            $paypalPayment = $payPalService->executePayment($_GET['paymentId'], $_GET['PayerID']);

            // Store Payment Details on the Database
            $p = json_decode($paypalPayment->toJSON());
            if (isset($p->id, $p->payer, $p->transactions)) {
                $this->payment = new PaypalPayment();
                $this->payment->form_id = $form_id;
                $this->payment->paypal_id = $model->id;
                $this->payment->submission_id = $submission_id;
                $this->payment->payment_id = $p->id;
                $this->payment->payment_state = $p->state;
                $this->payment->payment_method = $p->payer->payment_method;
                $this->payment->payer_status = $p->payer->status;
                $this->payment->payer_email = $p->payer->payer_info->email;
                $this->payment->payer_first_name = $p->payer->payer_info->first_name;
                $this->payment->payer_last_name = $p->payer->payer_info->last_name;
                $this->payment->payer_id = $p->payer->payer_info->payer_id;
                $this->payment->payer_country_code = $p->payer->payer_info->country_code;
                foreach ($p->transactions as $transaction) {
                    $this->payment->transaction_total = $transaction->amount->total;
                    $this->payment->transaction_currency = $transaction->amount->currency;
                    $this->payment->transaction_merchant_id = $transaction->payee->merchant_id;
                    $this->payment->payee_email = $transaction->payee->email;
                }
                $this->payment->save(false);

                $this->trigger(EventHelper::EVENT_PAYPAL_PAYMENT_RECEIVED);
            }

            // Redirect to another page
            return $this->redirect($model->return_url . (strpos($model->return_url, '?') === false ? '?' : '&') . http_build_query(['success' => is_object($this->payment)]));
        }
    }
}