<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal\controllers;

use app\components\User;
use app\modules\addons\modules\paypal\helpers\Currency;
use Yii;
use yii\filters\AccessControl;
use yii\web\Controller;
use yii\filters\VerbFilter;
use app\helpers\ArrayHelper;
use yii\web\NotFoundHttpException;
use app\models\Form;
use app\modules\addons\modules\paypal\models\Paypal;
use app\modules\addons\modules\paypal\models\PaypalItem;
use app\modules\addons\modules\paypal\models\PaypalSearch;
use app\modules\addons\modules\paypal\models\PaypalPaymentSearch;

/**
 * AdminController implements the CRUD actions for Paypal model.
 */
class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'currency-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } elseif ($postData = Yii::$app->request->post('Paypal')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'payments', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = Paypal::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\paypal\models\Paypal',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'currency-fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $currencies = [];
                    foreach (Currency::all() as $code => $name) {
                        array_push($currencies, [
                            'id' => $code,
                            'name' => $name,
                        ]);
                    }
                    $output = [
                        Yii::t('app', 'Currencies') => $currencies,
                    ];
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                $fieldsOutput = [];
                                foreach ($fields as $name => $label) {
                                    array_push($fieldsOutput, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                                $output[Yii::t('app', 'Fields')] = $fieldsOutput;
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
        ];
    }

    /**
     * Lists all Paypal models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new PaypalSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Lists all PaypalPayment models.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException
     */
    public function actionPayments($id)
    {
        $model = $this->findModel($id);
        $searchModel = new PaypalPaymentSearch();
        $searchModel->form_id = $model->form_id;
        $dataProvider = $searchModel->search([]);

        return $this->render('payments', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'model' => $model,
        ]);
    }

    /**
     * Displays a single Paypal model.
     * @param integer $id
     * @return mixed
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new Paypal model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Paypal();
        $itemModel = new PaypalItem();

        if ($model->load(Yii::$app->request->post()) && $model->validate()) {
            $model->items = Yii::$app->request->post('PaypalItem',[]);
            $model->save();
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The payment form has been successfully configured.')
            );
            return $this->redirect(['index']);
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'itemModel' => $itemModel,
            'forms' => $forms,
        ]);
    }

    /**
     * Updates an existing Paypal model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $itemModel = new PaypalItem();

        if ($model->load(Yii::$app->request->post()) && $model->validate()) {
            $model->items = Yii::$app->request->post('PaypalItem',[]);
            $model->save();
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The payment form has been successfully updated.')
            );
            return $this->redirect(['index']);
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'itemModel' => $itemModel,
            'forms' => $forms,
        ]);
    }

    /**
     * Enable / Disable multiple Payment Forms
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {

        $models = Paypal::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        } else {
            foreach ($models as $model) {
                $model->status = $status;
                $model->update();
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The selected items have been successfully updated.')
            );
            return $this->redirect(['index']);
        }
    }

    /**
     * Deletes an existing Paypal model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();
        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The payment form configuration has been successfully deleted.')
        );
        return $this->redirect(['index']);
    }

    /**
     * Finds the Paypal model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Paypal the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Paypal::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}