<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\paypal;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\EventHelper;
use app\helpers\Html;
use app\helpers\MailHelper;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\paypal\controllers\CheckController;
use app\modules\addons\modules\paypal\models\Paypal;
use app\modules\addons\modules\paypal\models\PaypalItem;
use app\modules\addons\modules\paypal\models\PaypalPayment;
use app\modules\addons\modules\paypal\services\PaypalService;
use Exception;
use Yii;
use yii\helpers\Url;
use yii\web\Response;

/**
 * Class Module
 * @package app\modules\addons\modules\paypal
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "paypal";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'Paypal' => Paypal::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                Form::EVENT_BEFORE_DELETE => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
            FormSubmission::class => [
                FormSubmission::EVENT_GET_SUBMISSION_DATA => [
                    [Module::class, 'onGetSubmissionData']
                ]
            ],
            EventHelper::class => [
                EventHelper::SUPPORTED_FORM_EVENTS => [
                    [self::class, 'onSupportedFormEvents']
                ]
            ],
            SubmissionHelper::class => [
                SubmissionHelper::GET_FIELDS_FOR_FIELD_MAPPING => [
                    [self::class, 'onGetFieldsForFieldMapping']
                ],
            ],
            CheckController::class => [
                EventHelper::EVENT_PAYPAL_PAYMENT_RECEIVED => [
                    [self::class, 'onPaymentReceived']
                ]
            ]
        ];
    }

    /**
     * @param $event
     * @throws Exception
     */
    public static function onPaymentReceived($event)
    {
        if (isset($event, $event->sender, $event->sender->payment)) {
            /**
             * Send Email Notification
             * Only when the required event occurs
             */
            /** @var PaypalPayment $payment */
            $payment = $event->sender->payment;
            if (in_array(EventHelper::STATUS_PAYPAL_PAYMENT_RECEIVED, $payment->form->formEmail->event)) {
                $submissionModel = FormSubmission::findOne(['id' => $payment->submission_id]);
                $formModel = $submissionModel->form;
                MailHelper::sendNotificationByEmail($formModel, $submissionModel);
            }
        }
    }

    /**
     * Event Handler
     * When Supported Form Events is executed
     *
     * @param $event
     */
    public static function onSupportedFormEvents($event)
    {
        $sender = $event->sender;
        if (!in_array(EventHelper::STATUS_PAYPAL_PAYMENT_RECEIVED, array_keys($sender->supportedFormEvents))) {
            $sender->supportedFormEvents[EventHelper::STATUS_PAYPAL_PAYMENT_RECEIVED] = Yii::t('app', 'Paypal payment received');
        }
        $event->sender = $sender;
    }

    /**
     * Event Handler
     * When Get Fields For Field Mapping is executed
     *
     * @param $event
     */
    public static function onGetFieldsForFieldMapping($event)
    {
        /** @var SubmissionHelper $sender */
        $sender = $event->sender;
        $fieldsForFieldMapping = $sender->fieldsForFieldMapping;
        $labels = (new PaypalPayment)->getLabelsForFieldMapping();
        unset($labels['id'], $labels['form_id'], $labels['submission_id'], $labels['created_at'], $labels['updated_at']);
        $sender->fieldsForFieldMapping = array_merge($fieldsForFieldMapping, $labels);
        $event->sender = $sender;
    }

    /**
     * Event Handler
     * When Get Submission Data is executed
     *
     * @param $event
     */
    public static function onGetSubmissionData($event)
    {
        /** @var FormSubmission $sender */
        $sender = $event->sender;
        $payment = PaypalPayment::findOne(['submission_id' => $sender->id]);
        $data = $sender->_data;
        if ($payment) {
            // Add Payment Data
            $sender->_data = array_merge($data, $payment->getValuesForFieldMapping($payment->fields()));
        } else {
            // Add Payment Link
            $sender->_data = array_merge($data, [
                'paypal_payment_link' => Url::to(['/addons/paypal/check/approval', 'sid' => $sender->hashId], true)
            ]);
        }
        $event->sender = $sender;
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = Paypal::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new Paypal();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new PaypalItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->paypal_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = Paypal::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {

        $result = false;

        $models = Paypal::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var \app\models\FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                try {

                    $success = true;
                    $message = !empty($model->message) ? $model->message :
                        Yii::t('app', 'Thank you for your business.');

                    // Create Paypal Payment and generate Approval Link
                    $service = new PaypalService($model->client_id, $model->client_secret, $model->mode);
                    $approvalLink = $service->createPayment($model, $formModel, $submissionModel, $submissionData);

                    // Send response
                    Yii::$app->getResponse()->on(Response::EVENT_BEFORE_SEND,
                        function($event) use ($success, $formModel, $message, $approvalLink) {
                            // Update response
                            /** @var \yii\web\Response $response */
                            $response = $event->sender;
                            $response->format = Response::FORMAT_JSON;
                            $response->data = array(
                                'action'  => 'submit',
                                'success' => $success,
                                'id' => $formModel->id,
                                'message' => $message,
                            );
                            // Redirect to PayPal
                            if ($approvalLink) {
                                $response->data['addon'] = [
                                    'redirectTo' => $approvalLink,
                                ];
                            }
                            $event->sender = $response;
                        });

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                    // PayPalConnectionException object
                    if (method_exists($e, 'getData')) {
                        // Send response
                        $message = Yii::t('app', 'PayPal payment failed.');
                        $error = json_decode($e->getData());
                        $details = [];
                        switch ($error->name) {
                            case 'VALIDATION_ERROR':
                                foreach ($error->details as $d)
                                {
                                    array_push($details, $d->issue);
                                }
                                $message .= Html::ul($details);
                                break;
                            default:
                                $message = Yii::t('app', 'An unexpected error has occurred. Please retry later.');
                        }
                        $response = Yii::$app->getResponse();
                        $response->format = Response::FORMAT_JSON;
                        $response->data = array(
                            'action'  => 'submit',
                            'success' => false,
                            'id' => $formModel->id,
                            'message' => $message,
                        );
                        $response->send();
                        exit;
                    }
                }
            }
        }

        return $result;
    }
}