<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 *
 * Based On Nexmo PHP Lib
 * @author Darren <darren@kiwiirc.com>
 * @see https://github.com/prawnsalad/Nexmo-PHP-lib
 * @license MIT
 */

namespace app\modules\addons\modules\nexmo\services;

use stdClass;

class NexmoService
{

    /**
     * @var bool Verify Nexmo SSL before sending any message
     */
    public $ssl = false;

    /**
     * @var string API URL
     */
    protected $apiUrl = 'https://rest.nexmo.com/sms/json';

    /**
     * @var string API Response
     */
    protected $apiResponse = '';

    /**
     * @var string API Key
     */
    private $apiKey = '';

    /**
     * @var string API Secret
     */
    private $apiSecret = '';

    /**
     * NexmoService constructor.
     *
     * @param $apiKey string API Key
     * @param $apiSecret string API Secret
     */
    function __construct($apiKey, $apiSecret) {
        $this->apiKey = $apiKey;
        $this->apiSecret = $apiSecret;
    }

    /**
     * Prepare new text message.
     *
     * If $unicode is not provided we will try to detect the
     * message type. Otherwise set to TRUE if you require
     * unicode characters.
     *
     * @param $to
     * @param $from
     * @param $message
     * @param null $unicode
     * @return array|bool|stdClass
     * @throws \Exception
     */
    public function sendSms($to, $from, $message, $unicode = null)
    {
        // Making sure strings are UTF-8 encoded
        if (!is_numeric($from) && !mb_check_encoding($from, 'UTF-8')) {
            throw new \Exception("$from needs to be a valid UTF-8 encoded string.");
        }

        if (!mb_check_encoding($message, 'UTF-8')) {
            throw new \Exception("$message needs to be a valid UTF-8 encoded string.");
        }

        if ($unicode === null) {
            $containsUnicode = max(array_map('ord', str_split($message))) > 127;
        } else {
            $containsUnicode = (bool)$unicode;
        }

        // Make sure $from is valid
        $from = $this->validateNumber($from);

        // URL Encode
        $from = urlencode( $from );
        $message = urlencode( $message );

        // Send away!
        $post = array(
            'from' => $from,
            'to' => $to,
            'text' => $message,
            'type' => $containsUnicode ? 'unicode' : 'text'
        );

        return $this->sendRequest($post);
    }

    /**
     * Validate an originator string
     *
     * If the originator ('from' field) is invalid, some networks may reject the network
     * whilst stinging you with the financial cost! While this cannot correct them, it
     * will try its best to correctly format them.
     *
     * @param $from
     * @return string
     */
    protected function validateNumber($from)
    {
        // Remove any invalid characters
        $ret = preg_replace('/[^a-zA-Z0-9]/', '', (string) $from);

        if(preg_match('/[a-zA-Z]/', $from)){
            // Alphanumeric format so make sure it's < 11 chars
            $ret = substr($ret, 0, 11);
        } else {
            // Numerical, remove any prepending '00'
            if(substr($ret, 0, 2) == '00'){
                $ret = substr($ret, 2);
                $ret = substr($ret, 0, 15);
            }
        }

        return (string) $ret;
    }

    /**
     * Prepare and send a new message.
     *
     * @param $data
     * @return array|bool|stdClass
     */
    protected function sendRequest($data)
    {
        // Build the post data
        $data = array_merge($data, array('username' => $this->apiKey, 'password' => $this->apiSecret));
        $post = '';
        foreach($data as $k => $v){
            $post .= "&$k=$v";
        }

        // If available, use CURL
        if (function_exists('curl_version')) {

            $to = curl_init( $this->apiUrl );
            curl_setopt($to, CURLOPT_POST, true);
            curl_setopt($to, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($to, CURLOPT_POSTFIELDS, $post);

            if (!$this->ssl) {
                curl_setopt($to, CURLOPT_SSL_VERIFYPEER, false);
            }

            $response = curl_exec($to);
            curl_close ($to);

        } elseif (ini_get('allow_url_fopen')) {

            // No CURL available so try the awesome file_get_contents
            $opts = array('http' =>
                array(
                    'method'  => 'POST',
                    'header'  => 'Content-type: application/x-www-form-urlencoded',
                    'content' => $post
                )
            );
            $context = stream_context_create($opts);
            $response = file_get_contents($this->apiUrl, false, $context);

        } else {

            // No way of sending a HTTP post
            return false;

        }

        return $this->parseResponse($response);
    }

    /**
     * Parse server response.
     *
     * @param $response
     * @return array|bool|stdClass
     */
    private function parseResponse($response)
    {
        $response = json_decode($response);

        // Copy the response data into an object, removing any '-' characters from the key
        /** @var \stdClass $responseObj */
        $responseObj = $this->normaliseKeys($response);

        if ($responseObj) {

            $this->apiResponse = $responseObj;
            // Find the total cost of this message
            $responseObj->cost = $total_cost = 0;

            if (isset($responseObj->messges) && is_array($responseObj->messages)) {
                foreach ($responseObj->messages as $msg) {
                    if (property_exists($msg, "messageprice")) {
                        $total_cost = $total_cost + (float) $msg->messageprice;
                    }
                }
                $responseObj->cost = $total_cost;
            }

            return $responseObj;

        } else {
            // A malformed response
            $this->apiResponse = [];
            return false;
        }
    }

    /**
     * Recursively normalise any key names in an object, removing unwanted characters
     *
     * @param $response
     * @return array|stdClass
     */
    private function normaliseKeys($response)
    {

        // Determine is working with a class or araay
        if ($response instanceof stdClass) {
            $newObj = new stdClass();
            $isObj = true;
        } else {
            $newObj = array();
            $isObj = false;
        }

        foreach($response as $key => $val){
            // If we come across another class/array, normalise it
            if ($val instanceof stdClass || is_array($val)) {
                $val = $this->normaliseKeys($val);
            }
            // Replace any unwanted characters in they key name
            if ($isObj) {
                $newObj->{str_replace('-', '', $key)} = $val;
            } else {
                $newObj[str_replace('-', '', $key)] = $val;
            }
        }

        return $newObj;
    }
}
