<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\neverbounce\models;

use app\components\behaviors\DateTrait;
use app\components\behaviors\RelationTrait;
use app\models\Form;
use app\models\User;
use Yii;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;

/**
 * This is the model class for table "{{%addon_neverbounce}}".
 *
 * @property integer $id
 * @property integer $form_id
 * @property integer $status
 * @property integer $role_address
 * @property integer $real_time
 * @property integer $disposable
 * @property string  $public_api_key
 * @property string  $private_api_key
 * @property integer $created_by
 * @property integer $updated_by
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Form $form
 * @property NeverBounceItem[] $items
 * @property User $author
 * @property User $lastEditor
 */
class NeverBounce extends \yii\db\ActiveRecord
{
    use RelationTrait, DateTrait;

    const ON = 1;
    const OFF = 0;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%addon_neverbounce}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['form_id', 'private_api_key'], 'required'],
            [['form_id'], 'unique', 'message' => Yii::t('app', 'This form already has a configuration.')],
            [['form_id', 'status', 'role_address', 'real_time', 'disposable'], 'integer'],
            [['private_api_key', 'public_api_key'], 'string', 'max' => 2555],
            [['public_api_key'], 'required', 'when' => function ($model) {
                return $model->real_time == 1;
            }, 'whenClient' => "function (attribute, value) {
                return $(\"input[name='NeverBounce[real_time]']:checked\").val() == '1';
            }"],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'form_id' => Yii::t('app', 'Form'),
            'status' => Yii::t('app', 'Status'),
            'private_api_key' => Yii::t('app', 'Private Api Key'),
            'public_api_key' => Yii::t('app', 'Public Api Key'),
            'role_address' => Yii::t('app', 'Role Address'),
            'real_time' => Yii::t('app', 'Real Time'),
            'disposable' => Yii::t('app', 'Disposable'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForm()
    {
        return $this->hasOne(Form::class, ['id' => 'form_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getItems()
    {
        return $this->hasMany(NeverBounceItem::class, ['form_id' => 'form_id']);
    }

    public function setItems($value)
    {
        $this->loadRelated('items', $value);
    }

    /**
     * @inheritdoc
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            // Delete all Items related to this Form
            NeverBounceItem::deleteAll(["form_id" => $this->form_id]);
            return true;
        } else {
            return false;
        }
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }
}