<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\neverbounce;

use app\controllers\AjaxController;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\neverbounce\models\NeverBounce;
use app\modules\addons\modules\neverbounce\models\NeverBounceItem;
use app\modules\addons\modules\neverbounce\services\NeverBounceService;
use Yii;
use yii\web\Response;
use yii\web\View;

/**
 * Class Module
 * @package app\modules\addons\modules\neverbounce
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "neverbounce";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';
    public static $errors = [];

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'NeverBounce' => NeverBounce::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            FormSubmission::class => [
                'afterValidate' => [
                    [Module::class, 'onFormSubmissionAfterValidate']
                ],
            ],
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormBeforeDelete']
                ]
            ],
            View::class => [
                'afterRender' => [
                    [Module::class, 'onViewAfterRender']
                ],
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = NeverBounce::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new NeverBounce();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new NeverBounceItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a Form model is deleted
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function onFormBeforeDelete($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $model = NeverBounce::findOne(['form_id' => $event->sender->id]);
            if ($model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a Form Submission model is validated
     *
	 * @param $event
	 * @throws \Exception
	 */
    public static function onFormSubmissionAfterValidate($event)
    {
        if (isset($event, $event->sender, $event->sender->form_id) && $event->sender instanceof FormSubmission) {

            $model = NeverBounce::findOne(['form_id' => $event->sender->form_id, 'status' => NeverBounce::ON]);

            if (isset($model, $model->form_id)) {

                $submissionData = $event->sender->data;
                self::$errors = $event->sender->getErrors();
                foreach ($model->items as $item) {
                    // Skip On Empty
                    if (isset($item->field_id) && !empty($submissionData[$item->field_id])) {

                        $value = $submissionData[$item->field_id];
                        $value = is_array($value) ? implode(',', $value) : $value;

                        self::validate($value, $item, $submissionData, $model->private_api_key);
                    }
                }
                $event->sender->clearErrors();
                $event->sender->addErrors(self::$errors);
            }
        }
    }

    /**
     * Validate
     *
     * @param string $value
     * @param NeverBounceItem $item
     * @param array $submissionData
     * @param string $apiKey
     * @throws \Exception
     */
    public static function validate($value, $item, $submissionData, $apiKey)
    {
        $neverBounceService = new NeverBounceService($apiKey);
        if (!$neverBounceService->validateEmail($value)) {
            $message = Yii::t('app', '{attribute} is not a valid email address.');
            if (!empty($item->field_message)) {
                $message = $item->field_message;
            }
            self::addErrorMessage($item, $message);
        }
    }

    /**
     * Add Error Message
     *
     * @param NeverBounceItem $item
     * @param string $message
     */
    public static function addErrorMessage($item, $message)
    {
        $errors = self::$errors;
        $errors[$item->field_id] = [$message];
        self::$errors = $errors;
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "app/embed" ) {

            $formModel = $event->sender->context->getFormModel();
            $model = NeverBounce::findOne(['form_id' => $formModel->id, 'status' => 1]);

            if ($model && $model->real_time && !empty($model->public_api_key)) {

                $selector = [];
                foreach ($model->items as $item) {
                    if (isset($item->field_id, $item->field_message)) {
                        $selector[] = '#'.$item->field_id;
                    }
                }
                $selector = implode(",", $selector);

                // Add client side validation only to app/embed view
                $code = <<<CODE

<script type="text/javascript">
    _NBSettings = {
        apiKey: '{$model->public_api_key}',
        ajaxMode: false,
        hiddenField: false,
        displayPoweredBy: false,
        autoFieldHookup: false,
        feedback: true
    };
</script>
<script type="text/javascript" src="https://cdn.neverbounce.com/widget/dist/NeverBounce.js"></script>
<script>
window.onload = (event) => {

    $('{$selector}').each(function( index ) {
        if ($(this).is(':visible')) {
             _nb.fields.registerListener(this, true);
        }
    });

    $('{$selector}').on('ef.show', function(e) {
        if ($(this).attr('type') === 'email') {
            _nb.fields.registerListener(this, true);        
        }
    })
    
    $('{$selector}').on('ef.hide', function() {
        if ($(this).attr('type') === 'email') {
            _nb.fields.unregisterListener(this);
        }
    })

};
</script>
</body>
CODE;
                $content = $event->output;
                $event->output =  str_replace("</body>", $code, $content);
            }
        }
    }
}