<?php

namespace app\modules\addons\modules\microsoft_dynamics\services;

use Exception;
use Yii;

class MicrosoftDynamicsService
{

    /** @var string Instance URI */
    protected $instanceURI = "https://example.crm.dynamics.com/";

    /** @var string Application ID */
    protected $applicationID = "00000000-0000-0000-0000-000000000000";

    /** @var string Application Secret */
    protected $applicationSecret = "Application Secret";

    /** @var string WebApi */
    protected $client = null;

    /** @var string Metadata */
    protected $metadata = null;

    /**
     * MicrosoftDynamicsService constructor.
     * @param $instanceURI
     * @param $applicationID
     * @param $applicationSecret
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function __construct($instanceURI, $applicationID, $applicationSecret)
    {
        if (empty($instanceURI) || empty($applicationID) || empty($applicationSecret)) {
            throw new Exception('A required parameter is empty.');
        }

        require_once(Yii::getAlias('@addons/microsoft_dynamics/services/vendor/autoload.php'));

        $this->client = \AlexaCRM\WebAPI\ClientFactory::createOnlineClient($instanceURI, $applicationID, $applicationSecret);
        $this->metadata = new \AlexaCRM\WebAPI\MetadataRegistry($this->client);
    }

    /**
     * @param string $entityName Eg. 'contact'
     * @param array $values
     * @return string Entity Id
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function createEntity($entityName, $values)
    {
        $entity = new \AlexaCRM\Xrm\Entity($entityName);
        foreach ($values as $attributeName => $value) {
            $entity[$attributeName] = $value;
        }
        return $this->client->Create($entity);
    }

    /**
     * @param string $entityName Eg. 'contact'
     * @param string $entityId
     * @param array $values
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function updateEntity($entityName, $entityId, $values)
    {
        $record = new \AlexaCRM\Xrm\Entity($entityName, $entityId);
        foreach ($values as $attributeName => $value) {
            $record[$attributeName] = $value;
        }
        $this->client->Update($record);
    }

    /**
     * Find Entities
     *
     * @param string $entityName Eg. 'contact'
     * @param array $attributes Eg. ['emailaddress1' => 'john@doe.com']
     * @param array|boolean $columns Eg. ['fullname', 'emailaddress1']
     * @return \AlexaCRM\Xrm\EntityCollection|bool
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function findEntities($entityName, $attributes = [], $columns = true)
    {
        $query = new \AlexaCRM\Xrm\Query\QueryByAttribute($entityName);
        foreach ($attributes as $attribute => $value) {
            $query->AddAttributeValue($attribute, $value);
        }
        $query->ColumnSet = new \AlexaCRM\Xrm\ColumnSet($columns);
        $collection = $this->client->RetrieveMultiple($query);

        if (count($collection->Entities) > 0) {
            return $collection;
        }

        return false;
    }

    /**
     * Find Entity By Attribute Value
     *
     * @param string $entityName Eg. 'contact'
     * @param array $attributes Eg. ['emailaddress1' => 'john@doe.com']
     * @param array|boolean $columns Eg. ['fullname', 'emailaddress1']
     * @return false|AlexaCRM\Xrm\Entity
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function findEntity($entityName, $attributes = [], $columns = true)
    {
        $collection = $this->findEntities($entityName, $attributes, $columns);

        if (!empty($collection->Entities)) {
            return $collection->current();
        }

        return false;
    }

    /**
     * Get Entity Reference
     *
     * @param $entityName
     * @param $entityId
     * @return \AlexaCRM\Xrm\EntityReference
     */
    public function getEntityReference($entityName, $entityId)
    {
        return new \AlexaCRM\Xrm\EntityReference($entityName, $entityId);
    }

    /**
     * Qualifies a lead and create account, contact, and opportunity records
     * that are linked to the originating lead record.
     *
     * @param $leadId
     * @param bool $createAccount
     * @param bool $createContact
     * @param bool $createOpportunity
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OData\ODataException
     * @throws \AlexaCRM\WebAPI\OData\TransportException
     */
    public function qualifyLead($leadId, $createAccount = false, $createContact = false, $createOpportunity = false)
    {
        return $this->client->getClient()->executeAction(
            'leads('.$leadId.')/Microsoft.Dynamics.CRM.QualifyLead', [
                "CreateAccount" => $createAccount,
                "CreateContact" => $createContact,
                "CreateOpportunity" => $createOpportunity,
                "Status" => 3, // Qualified
            ]);
    }

    /**
     * Associate Entities
     *
     * @param string $entityName Eg. 'opportunity'
     * @param $entityId
     * @param string $schemaName Eg. 'opportunity_customer_accounts', 'contact_customer_accounts', 'incident_customer_accounts', 'lead_customer_accounts'
     * @param string $referenceEntityName 'account'
     * @param $referenceEntityId
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function associateEntities($entityName, $entityId, $schemaName, $referenceEntityName, $referenceEntityId)
    {
        $this->client->Associate($entityName, $entityId,
            new \AlexaCRM\Xrm\Relationship($schemaName),
            [new \AlexaCRM\Xrm\EntityReference($referenceEntityName, $referenceEntityId)]
        );
    }

    /**
     * Get Marketing Lists
     *
     * @return array
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function getMarketingLists()
    {
        $collection = $this->findEntities('list', [], ['listid', 'listname']);

        $marketingLists = [];

        if (!empty($collection->Entities)) {
            foreach ($collection->Entities as $entity) {
                if (!empty($entity->Attributes)) {
                    if (!empty($entity->Attributes['listid']) && !empty($entity->Attributes['listname'])) {
                        $marketingLists[$entity->Attributes['listid']] = $entity->Attributes['listname'];
                    }
                }
            }
        }

        return $marketingLists;
    }

    /**
     * Add Member to Marketing List
     *
     * @param string $listId Marketing List ID
     * @param string $entityId Entity ID
     * @return mixed
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OData\ODataException
     * @throws \AlexaCRM\WebAPI\OData\TransportException
     */
    public function addMemberList($listId, $entityId)
    {
        return $this->client->getClient()->executeAction(
            'lists('.$listId.')/Microsoft.Dynamics.CRM.AddMemberList', [
            "EntityId" => $entityId,
        ]);
    }

    /**
     * Get Entity Attributes of Entity Name
     *
     * @param string $entityName Eg. 'contact'
     * @return array
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function getListFields($entityName)
    {
        $fields = array();
        $skip = array('accountid','ownerid','statecode','customerid','customeridtype','pricelevelid','ispricelocked','msdyn_accountmanagerid');
        $skipRequired = array('msdyn_psastatusreason','msdyn_psastate','msdyn_contractorganizationalunitid','msdyn_ordertype');
        $definition = $this->metadata->getDefinition($entityName);

        if (!empty($definition->Attributes)) {
            foreach ($definition->Attributes as $attribute) {
                if (isset($attribute->IsValidForUpdate) && $attribute->IsValidForUpdate == true
                    && !empty($attribute->DisplayName->UserLocalizedLabel->Label)) {

                    if (in_array($attribute->LogicalName, $skip)) {
                        continue;
                    }

                    $type = $attribute->AttributeTypeName->Value;

                    if ($type == 'DateTimeType' && isset($attribute->Format) && $attribute->Format == 'DateOnly') {
                        $type = 'DateType';
                    }

                    $field = array(
                        'name' => $attribute->LogicalName,
                        'type' => $type,
                        'label' => $attribute->DisplayName->UserLocalizedLabel->Label
                    );

                    if (isset($attribute->IsSearchable) && $attribute->AttributeType === 'StringType') {
                        $field['search'] = 'true';
                    }

                    if (isset($attribute->IsCustomAttribute) && $attribute->IsCustomAttribute === true) {
                        $field['is_custom'] = '1';
                    }

                    if ($attribute->RequiredLevel->Value != 'None' && !in_array($attribute->LogicalName, $skipRequired)) {
                        $field['required'] = 'true';
                    }

                    if (!empty($attribute->MaxLength)) {
                        $field['maxlength'] = $attribute->MaxLength;
                    }

                    if (!empty($attribute->Targets)) {
                        if ($attribute->AttributeType == 'Lookup') {
                            $field['related_to'] = $attribute->Targets[0];
                            $field['schema_name'] = $attribute->LogicalName;  //LogicalName , SchemaName
                            // $field['type'] = $field['related_to'].'-'.$field['type'];
                        } else if (count($attribute->Targets)) {
                            $name = $field['name'];
                            $label = $field['label'];
                            foreach ($attribute->Targets as $target) {
                                $field['related_to'] = $target;
                                $field['name'] = $name . '_' . $target;
                                $field['label'] = $label . '-' . ucfirst($target);
                                $fields[$field['name']] = $field;
                            }
                            continue;
                        }
                    }

                    $fields[$field['name']] = $field;

                    $options = []; $opt = []; $hint = []; $no = 0;

                    if (!empty($attribute->OptionSet->Options)) {
                        $options = $attribute->OptionSet->Options;
                    }

                    if (!empty($attribute->GlobalOptionSet->Options)) {
                        $options = $attribute->OptionSet->Options;
                    }

                    foreach ($options as $option) {
                        if (!empty($option->Label->UserLocalizedLabel->Label)) {
                            $opt[$option->Value] = $option->Label->UserLocalizedLabel->Label;
                            if ($no < 20) {
                                $hint[] = $option->Value . '=' . $opt[$option->Value];
                            }
                        }
                    }

                    $fields[$attribute->LogicalName]['options'] = $opt;
                    $fields[$attribute->LogicalName]['hint'] = implode(', ', $hint);
                }
            }
        }

        return $fields;
    }
}