<?php

namespace app\modules\addons\modules\microsoft_dynamics\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\helpers\Html;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\modules\microsoft_dynamics\helpers\Crm;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamics;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamicsField;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamicsLog;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamicsLogSearch;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamicsSearch;
use app\modules\addons\modules\microsoft_dynamics\Module;
use app\modules\addons\modules\microsoft_dynamics\services\MicrosoftDynamicsService;
use Exception;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\web\Response;

/**
 * AdminController implements the CRUD actions for MicrosoftDynamics model.
 */
class AdminController extends Controller
{
    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'list-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('MicrosoftDynamics')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'send', 'sending', 'process', 'resend', 'log', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = MicrosoftDynamics::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamics',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'list-fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($list_id, $params) {
                    $output = array();
                    $instanceUri = null;
                    $applicationId = null;
                    $applicationSecret = null;
                    $form_id = isset($params[0]) ? $params[0] : null;
                    $model = MicrosoftDynamics::findOne(['form_id' => $form_id]);
                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                            $instanceUri = $model->instance_uri;
                            $applicationId = $model->application_id;
                            $applicationSecret = $model->application_secret;
                        }
                    } else {
                        $session = Yii::$app->session;
                        $instanceUri = $session['microsoft_dynamics_instance_uri'];
                        $applicationId = $session['microsoft_dynamics_application_id'];
                        $applicationSecret = $session['microsoft_dynamics_application_secret'];
                    }
                    if ($instanceUri && $applicationId && $applicationSecret) {
                        $service = new MicrosoftDynamicsService($instanceUri, $applicationId, $applicationSecret);
                        $fields = $service->getListFields($list_id);
                        usort($fields, function($a, $b) {return isset($b['required']);});
                        foreach ($fields as $field) {
                            $label = $field['label'];
                            if (!empty($field['required'])) {
                                $label = $label . ' - ' . Yii::t('app', 'Required');
                            }
                            if (!empty($field['maxlength'])) {
                                $label = $label . ' - ' . Yii::t('app', 'Max Length') . ': ' . $field['maxlength'];
                            }
                            if (!empty($field['hint'])) {
                                $label = $label . ' - ' . Yii::t('app', 'Eg.') . ' ' . $field['hint'];
                            }
                            array_push($output, [
                                'id' => $field['name'],
                                'name' => $label,
                            ]);
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($listID, $params) {
                    if (isset($params[1]) && !empty($params[1])) {
                        return $params[1];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all MicrosoftDynamics models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new MicrosoftDynamicsSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single MicrosoftDynamics model.
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new MicrosoftDynamics model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $lists = Crm::lists();
        $marketingLists = [];
        $connected = false;

        $request = Yii::$app->request;

        $model = new MicrosoftDynamics();
        $fieldModel = new MicrosoftDynamicsField();

        try {
            if ($request->isPost && $model->load($request->post())) {

                $items = Yii::$app->request->post('MicrosoftDynamicsField',[]);

                $service = new MicrosoftDynamicsService($model->instance_uri, $model->application_id, $model->application_secret);

                if ($service) {

                    $connected = true;
                    $session = Yii::$app->session;
                    $session['microsoft_dynamics_instance_uri'] = $model->instance_uri;
                    $session['microsoft_dynamics_application_id'] = $model->application_id;
                    $session['microsoft_dynamics_application_secret'] = $model->application_secret;

                    // $marketingLists = $service->getMarketingLists();

                    if (!empty($items)) {
                        if ($model->validate()) {
                            $model->items = $items;
                            $model->save(false);
                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );

                            return $this->redirect(['index']);
                        } else {
                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify you configuration.')
                            );
                        }
                    }

                } else {
                    // Show error message
                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'Please enter a valid API information.')
                    );
                }
            }
        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', Yii::t('app', 'An error occurred connecting to your Microsoft Dynamics 365 account.'));
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
            'lists' => $lists,
            'marketingLists' => $marketingLists,
            'connected' => $connected,
        ]);
    }

    /**
     * Updates an existing MicrosoftDynamics model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param $id
     * @return string|\yii\web\Response
     * @throws Exception
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        $request = Yii::$app->request;
        $connected = false;
        $lists = Crm::lists();
        $marketingLists = [];

        try {

            $service = new MicrosoftDynamicsService($model->instance_uri, $model->application_id, $model->application_secret);

            if ($service) {

                $connected = true;
                // $marketingLists = $service->getMarketingLists();

                if ($request->isPost && $model->load($request->post())) {

                    $items = Yii::$app->request->post('MicrosoftDynamicsField',[]);

                    if (!empty($items)) {

                        if ($model->validate()) {

                            $model->items = $items;
                            $model->save(false);

                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );

                            return $this->redirect(['index']);

                        } else {

                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify you configuration.')
                            );
                        }
                    }
                }

            } else {
                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    Yii::t('app', 'Please enter a valid API information.')
                );
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', Yii::t('app', 'An error occurred connecting to your Microsoft Dynamics 365 account.'));
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'fieldModel' => new MicrosoftDynamicsField,
            'forms' => $forms,
            'lists' => $lists,
            'marketingLists' => $marketingLists,
            'connected' => $connected,
        ]);
    }

    /**
     * Enable / Disable multiple CleverReach models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = MicrosoftDynamics::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing MicrosoftDynamics model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Send form submissions to Microsoft Dynamics.
     * If configuration is successful, the browser will be redirected to the 'sending' page.
     *
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionSend($id)
    {

        $model = $this->findModel($id);
        $request = Yii::$app->request;
        $connected = false;

        try {

            /** @var Module $module */
            $module = $this->module;

            $service = new MicrosoftDynamicsService($model->instance_uri, $model->application_id, $model->application_secret);

            if ($service) {

                $connected = true;

                if ($request->isPost) {

                    if (Yii::$app->request->post('resent', false)) {

                        return $this->redirect(['sending', 'id' => $model->id, 'resent' => 1]);

                    }

                    return $this->redirect(['sending', 'id' => $model->id]);

                }

            } else {
                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                );

                return $this->redirect(['index']);
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', Yii::t('app', 'An error occurred connecting to your Microsoft Dynamics 365 account.'));
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('send', [
            'model' => $model,
            'forms' => $forms,
            'connected' => $connected,
        ]);
    }

    /**
     * UI to show the progress bar while sending form submissions to Microsoft Dynamics.
     * It shows a progress bar.
     *
     * @param $id
     * @param $resent
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionSending($id, $resent = 0)
    {
        $model = $this->findModel($id);

        return $this->render('sending', [
            'model' => $model,
            'resent' => $resent,
        ]);
    }

    /**
     * Submit form submissions to the Microsoft Dynamics using a batch process
     * @param int $id
     * @param int $resent
     * @param int $step
     * @param int $limit
     * @return array
     */
    public function actionProcess($id, $resent = 0, $step = 0, $limit = 1)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        try {

            $model = $this->findModel($id);

            if ($model->status === MicrosoftDynamics::ON) {

                // Set number of rows
                $nbRows = FormSubmission::find()
                    ->where(['form_id' => $model->form_id])
                    ->count();

                // Set offset
                $offset = $step * $limit;

                // Batch percentage
                $percentage = round(($offset + $limit) * 100 / $nbRows);

                // All rows have been processed ?
                $completed = ($offset + $limit) >= $nbRows;

                $rows = FormSubmission::find()
                    ->where(['form_id' => $model->form_id])
                    ->offset($offset)
                    ->limit($limit)
                    ->orderBy(['id' => SORT_DESC])
                    ->all();

                // Batch
                /** @var FormSubmission $row */
                foreach ($rows as $row) {

                    // Avoid sending previously sent entries
                    if (!$resent) {

                        $sent = MicrosoftDynamicsLog::findOne([
                            'microsoft_dynamics_id' => $model->id,
                            'submission_id' => $row->id,
                            'status' => 1,
                        ]);

                        if ($sent) {
                            continue;
                        }
                    }

                    /** @var Module $module */
                    $module = $this->module;

                    try {

                        // Submit data to Microsoft Dynamics according to the model configuration
                        $result = $module->submitData($model, $row->getSubmissionData());
                        // And log result
                        $module->logResult($model, $row, $result);

                    } catch (\Exception $e) {

                        // Log error
                        Yii::error($e);

                    }

                }

                $message = Yii::t('app',
                    'We have processed a batch of {0} form submissions.', [
                        Html::tag('span', count($rows), ['style' => 'font-weight: bold']),
                    ]);

                if ($completed) {
                    $message = Yii::t('app',
                        'Ready! We have processed {0} form submissions.', [
                            Html::tag('span', $nbRows, ['style' => 'font-weight: bold']),
                        ]);
                }

                return [
                    'completed' => $completed,
                    'step' => $completed ? 'done' : $step + 1,
                    'percentage' => $completed ? 100 : $percentage,
                    'limit' => $limit,
                    'message' => $message,
                ];

            }

        } catch (Exception $e) {
            Yii::error($e);
            return [
                'error' => 1,
            ];
        }

        return [
            'error' => 2,
        ];
    }

    /**
     * Re-Send a form submission to Microsoft Dynamics.
     * If configuration is successful, the browser will be redirected to the 'previous' page.
     *
     * @param int $id Microsoft Dynamics ID
     * @param int $sid Submission ID
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionResend($id, $sid)
    {

        $model = $this->findModel($id);

        try {

            $submission = FormSubmission::findOne(['id' => $sid]);

            // Verify if Submission ID was collected by the Form linked to this Microsoft Dynamics ID
            if ($model->form_id === $submission->form_id) {

                /** @var Module $module */
                $module = $this->module;

                $service = new MicrosoftDynamicsService($model->instance_uri, $model->application_id, $model->application_secret);

                if ($service) {

                    // Submit data to Microsoft Dynamics according to the model configuration
                    $result = $module->submitData($model, $submission->getSubmissionData());
                    // And log result
                    $module->logResult($model, $submission, $result);

                    if ($result) {

                        Yii::$app->getSession()->setFlash(
                            'success',
                            Yii::t('app','Form Submission was successfully re-sent to Microsoft Dynamics.')
                        );

                    } else {

                        Yii::$app->getSession()->setFlash(
                            'danger',
                            Yii::t('app', 'An error occurred re-sending the form submission to Microsoft Dynamics.')
                        );

                    }

                } else {

                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'Invalid Microsoft Dynamics settings found. Please verify your configuration.')
                    );
                }
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        $referrer = !empty(Yii::$app->request->referrer) ? Yii::$app->request->referrer : null;

        return $this->goBack($referrer);
    }

    /**
     * Lists all MicrosoftDynamicsLog models.
     *
     * @param $id
     *
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionLog($id)
    {
        $searchModel = new MicrosoftDynamicsLogSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('log', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Finds the MicrosoftDynamics model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return MicrosoftDynamics the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = MicrosoftDynamics::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
