<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\microsoft_dynamics;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamics;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamicsField;
use app\modules\addons\modules\microsoft_dynamics\models\MicrosoftDynamicsLog;
use app\modules\addons\modules\microsoft_dynamics\services\MicrosoftDynamicsService;
use Exception;
use Yii;
use yii\base\View;
use yii\helpers\Url;

/**
 * Class Module
 * @package app\modules\addons\modules\microsoft_dynamics
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "microsoft_dynamics";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';
    public static $rendered = false;

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'MicrosoftDynamics' => MicrosoftDynamics::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            View::class => [
                'afterRender' => [
                    [Module::class, 'onViewAfterRender']
                ],
            ],
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = MicrosoftDynamics::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new MicrosoftDynamics();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new MicrosoftDynamicsField();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->microsoft_dynamics_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = MicrosoftDynamics::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "form/submissions"
        ) {
            $formID = $event->sender->context->request->get("id");
            $formModel = Form::findOne(['id' => $formID]);

            // Flag to improve performance
            if (!self::$rendered && Yii::$app->user->can('configureFormsWithAddons', ['model' => $formModel])) {

                $models = MicrosoftDynamics::findAll(['form_id' => $formModel->id, 'status' => 1]);

                $content = $event->output;

                foreach ($models as $model) {
                    $content = str_replace(
                        '<i class="glyphicon glyphicon-print"></i>',
                        '<i class="glyphicon glyphicon-print"></i></a><a class="btn btn-info btn-microsoft_dynamics" href="#" data-href="' . Url::to(['/addons/microsoft_dynamics/admin/resend', 'id' => $model->id], true) . '" title="'.Yii::t('app', 'Send Form Submission to Microsoft Dynamics') . '"> <i class="glyphicon glyphicon-send"></i> ',
                        $content);
                }

                $content .= <<<EOT
<script>
window.addEventListener('load', function(){
  $('body').on('click', '.btn-microsoft_dynamics', function(event) {
        event.preventDefault();
        var hash = location.hash.substr(1),
        submission_id = hash.replace(/[^0-9]/g, ''),
        redirectWindow = window.open($(this).data('href') + '&sid=' + submission_id, '_blank');
        redirectWindow.location;
  });
});
</script>
EOT;

                $event->output =  $content;

                self::$rendered = true;
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {
        $result = false;

        $models = MicrosoftDynamics::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var \app\models\FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                try {

                    // Submit data according to the model configuration
                    $result = $this->submitData($model, $submissionData);
                    // Log result
                    $this->logResult($model, $submissionModel, $result);

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;
    }

    /**
     * Submit form submission to Microsoft Dynamics
     *
     * @param MicrosoftDynamics $model Microsoft Dynamics model
     * @param array $data Submission Data
     * @throws \AlexaCRM\WebAPI\OData\AuthenticationException
     * @throws \AlexaCRM\WebAPI\OData\ODataException
     * @throws \AlexaCRM\WebAPI\OData\TransportException
     * @throws \AlexaCRM\WebAPI\OrganizationException
     * @throws \AlexaCRM\WebAPI\ToolkitException
     */
    public function submitData($model, $data)
    {
        // Flag
        $entityID = null;

        $values = [];

        // Validate, format and set Field values
        foreach ($model->items as $item) {
            if (!empty($data[$item->form_field])) {
                // Default value
                $values[$item->list_field] = $data[$item->form_field];

                // Select List, Checkbox
                if (is_array($data[$item->form_field])) {
                    if (count($data[$item->form_field]) > 1) {
                        $values[$item->list_field] = implode(',', $data[$item->form_field]);
                    } else {
                        $values[$item->list_field] = $data[$item->form_field][0];
                    }
                }

                // Numbers
                if (is_numeric($data[$item->form_field])) {
                    if ((int)$data[$item->form_field] == (float)$data[$item->form_field]) {
                        $values[$item->list_field] = (int)$data[$item->form_field];
                    } else {
                        $values[$item->list_field] = (float)$data[$item->form_field];
                    }
                }
            }
        }

        if (!empty($values)) {

            $service = new MicrosoftDynamicsService(
                $model->instance_uri,
                $model->application_id,
                $model->application_secret
            );

            $customerEntity = null;
            $entity = null;
            $entityType = is_array($model->list_id) ? current($model->list_id) : $model->list_id;

            /**
             * Search Customer Entity (Contact or Lead)
             */
            $priorities = $model->search_priority === MicrosoftDynamics::SEARCH_PRIORITY_BY_CONTACT ? ['contact', 'lead'] : ['lead', 'contact'];
            if (!empty($data[$model->email_field])) {
                foreach ($priorities as $entityTypeWithPriority) {
                    $customerEntity = $service->findEntity($entityTypeWithPriority, [
                        $model->search_by => $data[$model->email_field]
                    ]);
                    if ($customerEntity && !empty($customerEntity->Id)) {
                        break;
                    }
                }
            }

            /**
             * Save entity
             */

            if ($model->action === MicrosoftDynamics::ACTION_SEARCH_AND_CREATE) {

                if ($entityType === 'lead') {
                    $entityID = $service->createEntity($entityType, $values);
                } elseif ($entityType === 'contact') {
                    $entityID = $service->createEntity($entityType, $values);
                } elseif ($entityType === 'account') {
                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {
                        // Create account and link to contact
                        $values['primarycontactid'] = $service->getEntityReference($customerEntity->LogicalName, $customerEntity->Id);
                        $entityID = $service->createEntity($entityType, $values);
                        $attributes = $customerEntity->Attributes;
                        $attributes['parentcustomerid'] = $service->getEntityReference($entityType, $entityID);
                        $service->updateEntity($customerEntity->LogicalName, $customerEntity->Id, $attributes);
                    } else {
                        // Only create
                        $entityID = $service->createEntity($entityType, $values);
                    }

                } elseif ($entityType === 'opportunity') {

                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {

                        $contactAttributes = $customerEntity->Attributes;
                        $contactReference = $service->getEntityReference($customerEntity->LogicalName, $customerEntity->Id);
                        // Link to contact
                        $values['parentcontactid'] = $contactReference;

                        // Link to account
                        if (!empty($contactAttributes['parentcustomerid'])
                            && !empty($contactAttributes['parentcustomerid']->LogicalName)
                            && $contactAttributes['parentcustomerid']->LogicalName === 'account') {
                            $accountID = $contactAttributes['parentcustomerid']->Id;
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to opportunity
                            $values['parentaccountid'] = $accountReference;
                        } elseif (!empty($data[$model->account_name_field])) {
                            $accountID = $service->createEntity('account', [
                                'primarycontactid' => $contactReference,
                                'name' => $data[$model->account_name_field],
                            ]);
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to contact
                            $contactAttributes['parentcustomerid'] = $accountReference;
                            $service->updateEntity('contact', $customerEntity->Id, $contactAttributes);
                            // Link account to opportunity
                            $values['parentaccountid'] = $accountReference;
                        }

                        // Create opportunity
                        $entityID = $service->createEntity($entityType, $values);

                    } else {

                        // Create contact & account
                        if (!empty($data[$model->first_name_field])
                            && !empty($data[$model->last_name_field])
                            && !empty($data[$model->email_field])) {

                            $contactAttributes = [
                                'firstname' => $data[$model->first_name_field],
                                'lastname' => $data[$model->last_name_field],
                                'emailaddress1' => $data[$model->email_field],
                            ];
                            $contactID = $service->createEntity('contact', $contactAttributes);
                            $contactReference = $service->getEntityReference('contact', $contactID);

                            // Create account and link to opportunity
                            if (!empty($data[$model->account_name_field])) {
                                $accountID = $service->createEntity('account', [
                                    'primarycontactid' => $contactReference,
                                    'name' => $data[$model->account_name_field],
                                ]);
                                $accountReference = $service->getEntityReference('account', $accountID);
                                // Link to contact
                                $contactAttributes['parentcustomerid'] = $accountReference;
                                $service->updateEntity('contact', $contactID, $contactAttributes);
                                // Link account to opportunity
                                $values['parentaccountid'] = $accountReference;
                            }

                            // Link contact to opportunity
                            $values['parentcontactid'] = $contactReference;
                        }

                        // Create opportunity
                        $entityID = $service->createEntity($entityType, $values);
                    }

                } elseif ($entityType === 'incident') {

                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {

                        $contactAttributes = $customerEntity->Attributes;
                        $contactReference = $service->getEntityReference($customerEntity->LogicalName, $customerEntity->Id);

                        // Link to Contact
                        $values['primarycontactid'] = $contactReference;

                        // Link to Account
                        if (!empty($contactAttributes['parentcustomerid'])
                            && !empty($contactAttributes['parentcustomerid']->LogicalName)
                            && $contactAttributes['parentcustomerid']->LogicalName === 'account') {
                            $accountID = $contactAttributes['parentcustomerid']->Id;
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to incident
                            $values['customerid'] = $accountReference;
                        } elseif (!empty($data[$model->account_name_field])) {
                            $accountID = $service->createEntity('account', [
                                'primarycontactid' => $contactReference,
                                'name' => $data[$model->account_name_field],
                            ]);
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to contact
                            $contactAttributes['parentcustomerid'] = $accountReference;
                            $service->updateEntity('contact', $customerEntity->Id, $contactAttributes);
                            // Link account to incident
                            $values['customerid'] = $accountReference;
                        }

                        // An account is a required attribute to create an incident
                        if (!empty($values['customerid'])) {
                            $entityID = $service->createEntity($entityType, $values);
                        }

                    } else {

                        // Create contact & account
                        if (!empty($data[$model->first_name_field])
                            && !empty($data[$model->last_name_field])
                            && !empty($data[$model->email_field])) {

                            $contactAttributes = [
                                'firstname' => $data[$model->first_name_field],
                                'lastname' => $data[$model->last_name_field],
                                'emailaddress1' => $data[$model->email_field],
                            ];
                            $contactID = $service->createEntity('contact', $contactAttributes);
                            $contactReference = $service->getEntityReference('contact', $contactID);

                            // Create account and link to incident
                            if (!empty($data[$model->account_name_field])) {
                                $accountID = $service->createEntity('account', [
                                    'primarycontactid' => $contactReference,
                                    'name' => $data[$model->account_name_field],
                                ]);
                                $accountReference = $service->getEntityReference('account', $accountID);
                                // Link to contact
                                $contactAttributes['parentcustomerid'] = $accountReference;
                                $service->updateEntity('contact', $contactID, $contactAttributes);
                                // Link account to incident
                                $values['customerid'] = $accountReference;
                            }

                            // Link contact to incident
                            $values['primarycontactid'] = $contactReference;
                        }

                        // An account is a required attribute to create an incident
                        if (!empty($values['customerid'])) {
                            $entityID = $service->createEntity($entityType, $values);
                        }
                    }
                }

            } elseif ($model->action === MicrosoftDynamics::ACTION_SEARCH_AND_UPDATE) {

                if ($entityType === 'lead') {
                    if ($customerEntity && !empty($customerEntity->LogicalName) && $customerEntity->LogicalName === 'lead') {
                        $service->updateEntity($entityType, $customerEntity->Id, $values);
                        $entityID = $customerEntity->Id;
                    }
                } elseif ($entityType === 'contact') {
                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {
                        $service->updateEntity($entityType, $customerEntity->Id, $values);
                        $entityID = $customerEntity->Id;
                    }
                } elseif ($entityType === 'account') {
                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {
                        // Find account by contact email
                        if (!empty($customerEntity->Attributes['parentcustomerid'])
                            && !empty($customerEntity->Attributes['parentcustomerid']->LogicalName)
                            && $customerEntity->Attributes['parentcustomerid']->LogicalName === 'account') {
                            $accountID = $customerEntity->Attributes['parentcustomerid']->Id;
                            $service->updateEntity($entityType, $accountID, $values);
                        }
                    }
                } elseif ($entityType === 'opportunity') {
                    // We can't update an opportunity
                } elseif ($entityType === 'incident') {
                    // We can't update an incident
                }

            } elseif ($model->action === MicrosoftDynamics::ACTION_SEARCH_AND_UPDATE_OR_CREATE) {

                if ($entityType === 'lead') {
                    if ($customerEntity && !empty($customerEntity->LogicalName) && $customerEntity->LogicalName === 'lead') {
                        $service->updateEntity($entityType, $customerEntity->Id, $values);
                        $entityID = $customerEntity->Id;
                    } else {
                        $entityID = $service->createEntity($entityType, $values);
                    }
                } elseif ($entityType === 'contact') {
                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {
                        $service->updateEntity($entityType, $customerEntity->Id, $values);
                        $entityID = $customerEntity->Id;
                    } else {
                        $entityID = $service->createEntity($entityType, $values);
                    }

                } elseif ($entityType === 'account') {

                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {

                        // Find account by contact email
                        if (!empty($customerEntity->Attributes['parentcustomerid'])
                            && !empty($customerEntity->Attributes['parentcustomerid']->LogicalName)
                            && $customerEntity->Attributes['parentcustomerid']->LogicalName === 'account') {
                            $accountID = $customerEntity->Attributes['parentcustomerid']->Id;
                            $service->updateEntity($entityType, $accountID, $values);
                        } else {
                            // Create account and link to contact
                            $values['primarycontactid'] = $service->getEntityReference($customerEntity->LogicalName, $customerEntity->Id);
                            $entityID = $service->createEntity($entityType, $values);
                            $attributes = $customerEntity->Attributes;
                            $attributes['parentcustomerid'] = $service->getEntityReference($entityType, $entityID);
                            $service->updateEntity($customerEntity->LogicalName, $customerEntity->Id, $attributes);
                        }

                    } else {

                        // Only create
                        $entityID = $service->createEntity($entityType, $values);

                    }
                } elseif ($entityType === 'opportunity') {

                    // We can't update an opportunity, only create one
                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {

                        $contactAttributes = $customerEntity->Attributes;
                        $contactReference = $service->getEntityReference($customerEntity->LogicalName, $customerEntity->Id);
                        // Link to contact
                        $values['parentcontactid'] = $contactReference;

                        // Link to account
                        if (!empty($contactAttributes['parentcustomerid'])
                            && !empty($contactAttributes['parentcustomerid']->LogicalName)
                            && $contactAttributes['parentcustomerid']->LogicalName === 'account') {
                            $accountID = $contactAttributes['parentcustomerid']->Id;
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to opportunity
                            $values['parentaccountid'] = $accountReference;
                        } elseif (!empty($data[$model->account_name_field])) {
                            $accountID = $service->createEntity('account', [
                                'primarycontactid' => $contactReference,
                                'name' => $data[$model->account_name_field],
                            ]);
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to contact
                            $contactAttributes['parentcustomerid'] = $accountReference;
                            $service->updateEntity('contact', $customerEntity->Id, $contactAttributes);
                            // Link account to opportunity
                            $values['parentaccountid'] = $accountReference;
                        }

                        // Create opportunity
                        $entityID = $service->createEntity($entityType, $values);

                    } else {

                        // Create contact & account
                        if (!empty($data[$model->first_name_field])
                            && !empty($data[$model->last_name_field])
                            && !empty($data[$model->email_field])) {

                            $contactAttributes = [
                                'firstname' => $data[$model->first_name_field],
                                'lastname' => $data[$model->last_name_field],
                                'emailaddress1' => $data[$model->email_field],
                            ];
                            $contactID = $service->createEntity('contact', $contactAttributes);
                            $contactReference = $service->getEntityReference('contact', $contactID);

                            // Create account and link to opportunity
                            if (!empty($data[$model->account_name_field])) {
                                $accountID = $service->createEntity('account', [
                                    'primarycontactid' => $contactReference,
                                    'name' => $data[$model->account_name_field],
                                ]);
                                $accountReference = $service->getEntityReference('account', $accountID);
                                // Link to contact
                                $contactAttributes['parentcustomerid'] = $accountReference;
                                $service->updateEntity('contact', $contactID, $contactAttributes);
                                // Link account to opportunity
                                $values['parentaccountid'] = $accountReference;
                            }

                            // Link contact to opportunity
                            $values['parentcontactid'] = $contactReference;
                        }

                        // Create opportunity
                        $entityID = $service->createEntity($entityType, $values);
                    }

                } elseif ($entityType === 'incident') {

                    // We can't update an incident, only create one
                    if ($customerEntity
                        && !empty($customerEntity->LogicalName)
                        && $customerEntity->LogicalName === 'contact') {

                        $contactAttributes = $customerEntity->Attributes;
                        $contactReference = $service->getEntityReference($customerEntity->LogicalName, $customerEntity->Id);
                        // Link to Contact
                        $values['primarycontactid'] = $contactReference;

                        // Link to Account
                        if (!empty($contactAttributes['parentcustomerid'])
                            && !empty($contactAttributes['parentcustomerid']->LogicalName)
                            && $contactAttributes['parentcustomerid']->LogicalName === 'account') {

                            $accountID = $contactAttributes['parentcustomerid']->Id;
                            $accountReference = $service->getEntityReference('account', $accountID);
                            // Link account to incident
                            $values['customerid'] = $accountReference;

                        } elseif (!empty($data[$model->account_name_field])) {

                            $accountID = $service->createEntity('account', [
                                'primarycontactid' => $contactReference,
                                'name' => $data[$model->account_name_field],
                            ]);
                            $accountReference = $service->getEntityReference('account', $accountID);

                            // Link account to contact
                            $contactAttributes['parentcustomerid'] = $accountReference;
                            $service->updateEntity('contact', $customerEntity->Id, $contactAttributes);

                            // Link account to incident
                            $values['customerid'] = $accountReference;
                        }

                        // An account is a required attribute to create an incident
                        if (!empty($values['customerid'])) {
                            $entityID = $service->createEntity($entityType, $values);
                        }

                    } else {

                        // Create contact & account
                        if (!empty($data[$model->first_name_field])
                            && !empty($data[$model->last_name_field])
                            && !empty($data[$model->email_field])) {

                            $contactAttributes = [
                                'firstname' => $data[$model->first_name_field],
                                'lastname' => $data[$model->last_name_field],
                                'emailaddress1' => $data[$model->email_field],
                            ];

                            $contactID = $service->createEntity('contact', $contactAttributes);
                            $contactReference = $service->getEntityReference('contact', $contactID);

                            // Link to contact
                            $values['primarycontactid'] = $contactReference;

                            // Create account and link to incident
                            if (!empty($data[$model->account_name_field])) {
                                $accountID = $service->createEntity('account', [
                                    'primarycontactid' => $contactReference,
                                    'name' => $data[$model->account_name_field],
                                ]);
                                $accountReference = $service->getEntityReference('account', $accountID);
                                // Link to contact
                                $contactAttributes['parentcustomerid'] = $accountReference;
                                $service->updateEntity('contact', $contactID, $contactAttributes);
                                // Link account to incident
                                $values['customerid'] = $accountReference;
                            }
                        }

                        // An account is a required attribute to create an incident
                        if (!empty($values['customerid'])) {
                            $entityID = $service->createEntity($entityType, $values);
                        }
                    }
                }
            }

            /**
             * Qualify lead
             */

            if (!empty($entityID) && $model->qualify_lead === MicrosoftDynamics::ON) {
                if ($entityType === 'lead') {
                    if ($entity && !empty($entity->Id)) { // When a lead is updated
                        // Verify status code
                        if ($entity
                            && !empty($entity->Attributes)
                            && !empty($entity->Attributes['statuscode'])
                            && $entity->Attributes['statuscode'] < 3) {
                            $service->qualifyLead(
                                $entityID,
                                (boolean) $model->create_account,
                                (boolean) $model->create_contact,
                                (boolean) $model->create_opportunity
                            );
                        }
                    } else { // When a lead is created
                        $service->qualifyLead(
                            $entityID,
                            (boolean) $model->create_account,
                            (boolean) $model->create_contact,
                            (boolean) $model->create_opportunity
                        );
                    }
                }
            }

            /**
             * Add Entity To Marketing List
             */

            // if (!empty($entityID) && $model->has_marketing_list && !empty($model->marketing_list)) {
            //    if (in_array($entityType, ['account', 'contact', 'lead'])) {
            //        $service->addMemberList($model->marketing_list, $entityID);
            //    }
            // }

        }

        return $entityID;
    }

    /**
     * @param MicrosoftDynamics $model
     * @param FormSubmission $submissionModel
     * @param string $result
     */
    public function logResult($model, $submissionModel, $result)
    {
        // Update / Insert Log
        $log = MicrosoftDynamicsLog::findOne([
            'microsoft_dynamics_id' => $model->id,
            'form_id' => $model->form_id,
            'submission_id' => $submissionModel->id]);

        if (!$log) {
            $log = new MicrosoftDynamicsLog();
        }

        $log->microsoft_dynamics_id = $model->id;
        $log->form_id = $model->form_id;
        $log->submission_id = $submissionModel->id;
        $log->log = json_encode(['entity_id' => $result]);
        $log->status = !empty($result);
        $log->save(false);
    }
}