<?php

namespace app\modules\addons\modules\mailerlite\services;

use Exception;
use yii\helpers\Json;

class MailerLiteService
{

    /** @var string Api Key */
    protected $apiKey;

    /** @var string Base Url */
    protected $baseUrl = "https://api.mailerlite.com/api/v2";

    /** @var string Response Format */
    protected $returnFormat = "json";

    /** @var bool Check Header */
    protected $checkHeader = true;

    /** @var bool Throw Exceptions */
    protected $throwExceptions = true;

    /** @var bool Header */
    protected $header = false;

    /** @var bool Error */
    protected $error = false;

    /**
     * MailerLiteService constructor.
     * @param $apiKey
     * @throws Exception
     */
    public function __construct($apiKey)
    {
        if (empty($apiKey)) {
            throw new Exception('The api key is empty');
        }
        $this->apiKey = $apiKey;
    }

    /**
     * Add or Edit Subscriber
     *
     * @param int $groupId
     * @param array $contact
     * @return array|false
     * @throws Exception
     */
    public function syncContact($groupId, $contact)
    {
        $contact = (is_array($contact)) ? Json::encode($contact) : $contact;
        $options = [
            CURLOPT_URL => $this->baseUrl . "/groups/{$groupId}/subscribers",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => false,
            CURLOPT_HTTPHEADER => array(
                "X-MailerLite-ApiKey: $this->apiKey",
                "cache-control: no-cache",
                "content-type: application/json"
            ),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => $contact,
            CURLINFO_HEADER_OUT => true,
        ];
        return (array) $this->request($options);
    }

    /**
     * Get Subscriber Lists
     *
     * @return array
     * @throws Exception
     */
    public function getSubscriberLists()
    {
        $options = [
            CURLOPT_URL => $this->baseUrl . "/groups",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => false,
            CURLOPT_HTTPHEADER => array(
                "X-MailerLite-ApiKey: $this->apiKey",
                "cache-control: no-cache",
                "content-type: application/json"
            ),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLINFO_HEADER_OUT => true,
        ];

        $listCollection  =  (array) $this->request($options);

        $lists = [];
        if (is_array($listCollection) && !empty($listCollection)) {
            foreach($listCollection as $list) {
                $list = (array) $list;
                if (isset($list['id'], $list['name'])) {
                    $lists[$list['id']] = $list['name'];
                }
            }
        }

        return $lists;
    }

    /**
     * Get Custom Fields
     *
     * @return array
     * @throws Exception
     */
    public function getCustomFields()
    {
        $options = [
            CURLOPT_URL => $this->baseUrl . "/fields",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => false,
            CURLOPT_HTTPHEADER => array(
                "X-MailerLite-ApiKey: $this->apiKey",
                "cache-control: no-cache",
                "content-type: application/json"
            ),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLINFO_HEADER_OUT => true,
        ];

        $customFieldCollection  =  (array) $this->request($options);

        $fields = [];
        if (is_array($customFieldCollection) && !empty($customFieldCollection)) {
            foreach($customFieldCollection as $customField) {
                $customField = (array) $customField;
                if (isset($customField['key'], $customField['title']) && !($customField['key'] === 'email')) {
                    $fields[$customField['key']] = $customField['title'];
                }
            }
        }
        return $fields;
    }

    /**
     * Return formatted response according to settings
     *
     * @param $in
     * @param bool $header
     * @return mixed|null
     * @throws Exception
     */
    protected function response($in, $header = false)
    {
        $this->header = $header;

        if ($this->checkHeader && isset($header["http_code"])) {
            if ($header["http_code"] < 200 || $header["http_code"] >= 300) {
                //error!?
                $this->error = $in;
                $message = var_export($in, true);
                if ($tmp = json_decode($in)) {
                    if (isset($tmp->error->message)) {
                        $message = $tmp->error->message;
                    }
                }
                if ($this->throwExceptions) {
                    throw new \Exception('' . $header["http_code"] . ';' . $message);
                }
                $in = null;

            }

        }

        switch ($this->returnFormat) {
            case 'json':
                return json_decode($in);
                break;

            default:
                return $in;
                break;
        }
        return $in;
    }

    /**
     * @param $options
     * @return mixed|null
     * @throws Exception
     */
    protected function request($options)
    {
        $curl = curl_init();
        curl_setopt_array($curl, $options);
        $response = curl_exec($curl);
        $info =  curl_getinfo($curl);
        curl_close($curl);

        return $this->response($response, $info);
    }

}