<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\mailchimp\services;

use Yii;
use Exception;

class MailchimpService
{
    /** @var  string API Key */
    protected $apiKey;

    /**
     * MailchimpService constructor.
     * @param $apiKey
     */
    public function __construct($apiKey)
    {
        spl_autoload_register(function(){
            require_once(Yii::getAlias('@addons/mailchimp/mailchimp_api/MailChimp.php'));
        });

        $this->apiKey = $apiKey;
    }

    /**
     * Get Subscriber Lists
     *
     * @return array
     */
    public function getSubscriberLists()
    {

        $MailChimp = new \MailChimp($this->apiKey);
        $listCollection = $MailChimp->get('lists');

        $lists = [];

        if (is_array($listCollection) && !empty($listCollection) && isset($listCollection['lists']) && is_array($listCollection['lists'])) {
            foreach($listCollection['lists'] as $list) {
                $lists[$list['id']] = $list['name'];
            }
        }

        return $lists;
    }

    /**
     * Get Custom Fields
     *
     * @param $listID
     * @return array
     */
    public function getCustomFields($listID)
    {
        $MailChimp = new \MailChimp($this->apiKey);

        $customFieldUrl = "/lists/{$listID}/merge-fields";
        $customFieldCollection = $MailChimp->get($customFieldUrl);
        $fields = [];

        foreach($customFieldCollection['merge_fields'] as $customField ) {
            array_push($fields, [
                'id' => $customField['tag'],
                'name' => $customField['name'],
            ]);
        }

        return $fields;
    }

    /**
     * Get Interest List
     * Group of interests
     *
     * @param $listID
     * @return array
     */
    public function getInterestList($listID)
    {
        $MailChimp = new \MailChimp($this->apiKey);

        $interestCategoriesUrl = "/lists/{$listID}/interest-categories";
        $interestCategoriesCollection = $MailChimp->get($interestCategoriesUrl);

        $interestList = [];
        foreach ($interestCategoriesCollection['categories'] as $group) {
            // we can skip hidden interests
            if ($group['type'] === 'hidden') {
                continue;
            }

            // connect to API to get interests from each category
            $interestsUrl = "/lists/{$listID}/interest-categories/{$group['id']}/interests";
            $interests = $MailChimp->get($interestsUrl);

            if ($interests['total_items'] > 0) {
                switch ($group['type']) {
                    case 'checkboxes': {
                        foreach ($interests['interests'] as $interest) {
                            $interestList[$interest['id']] = $interest['name'];
                        }
                        break;
                    }
                    case 'radio': {
                        foreach ($interests['interests'] as $interest) {
                            $interestList[$interest['id']] = $interest['name'];
                        }
                        break;
                    }
                    case 'dropdown': {
                        foreach ($interests['interests'] as $interest) {
                            $interestList[$interest['id']] = $interest['name'];
                        }
                        break;
                    }
                    default:
                        break;
                };
            }
        }

        return $interestList;
    }

    /**
     * Add Subscriber to a List
     *
     * @param $listID
     * @param $params
     * @return array|false
     * @throws Exception
     */
    public function addSubscriber($listID, $params)
    {
        $MailChimp = new \MailChimp($this->apiKey);
        $result = $MailChimp->post("lists/$listID/members", $params);

        if (!$MailChimp->success()) {
            throw new Exception($MailChimp->getLastError());
        }

        return $result;
    }

    /**
     * Update Subscriber
     *
     * @param $listID
     * @param $subscriberHash
     * @param $params
     * @return array|false
     * @throws Exception
     */
    public function updateSubscriber($listID, $subscriberHash, $params)
    {
        $MailChimp = new \MailChimp($this->apiKey);
        $result = $MailChimp->patch("lists/{$listID}/members/{$subscriberHash}", $params);

        if (!$MailChimp->success()) {
            throw new Exception($MailChimp->getLastError());
        }

        return $result;
    }

    /**
     * Get Subscriber Hash
     *
     * @param $email
     * @return string
     */
    public function getSubscriberHash($email)
    {
        $MailChimp = new \MailChimp($this->apiKey);
        $result = $MailChimp->subscriberHash($email);

        return $result;
    }
}