<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\limit_choices\models;

use app\components\behaviors\DateTrait;
use app\models\User;
use Yii;
use app\models\Form;
use app\components\behaviors\RelationTrait;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;

/**
 * This is the model class for table "{{%addon_limit_choices}}".
 *
 * @property integer $id
 * @property integer $form_id
 * @property integer $status
 * @property string $unavailable
 * @property integer $remaining_quantity
 * @property string  $remaining_quantity_text
 * @property string  $remaining_quantity_error
 * @property integer $created_by
 * @property integer $updated_by
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Form $form
 * @property User $author
 * @property User $lastEditor
 * @property LimitChoicesItem[] $items
 */
class LimitChoices extends \yii\db\ActiveRecord
{
    use RelationTrait, DateTrait;

    const OFF = 0;
    const ON = 1;

    const UNAVAILABLE_DISABLE = 'disable';
    const UNAVAILABLE_HIDE = 'hide';
    const UNAVAILABLE_DISABLE_FIELD = 'disable_field';
    const UNAVAILABLE_DISABLE_FORM = 'disable_form';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%addon_limit_choices}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['form_id'], 'required'],
            [['form_id'], 'unique', 'message' => Yii::t('app', 'This form already has a configuration.')],
            [['form_id', 'status', 'remaining_quantity'], 'integer'],
            [['unavailable', 'remaining_quantity_text', 'remaining_quantity_error'], 'string'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'form_id' => Yii::t('app', 'Form'),
            'status' => Yii::t('app', 'Status'),
            'unavailable' => Yii::t('app', 'When a Choice is Unavailable'),
            'remaining_quantity' => Yii::t('app', 'Show Remaining Quantity'),
            'remaining_quantity_text' => Yii::t('app', 'Available Choice Message'),
            'remaining_quantity_error' => Yii::t('app', 'Unavailable Choice Message'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForm()
    {
        return $this->hasOne(Form::class, ['id' => 'form_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getItems()
    {
        return $this->hasMany(LimitChoicesItem::class, ['limit_choices_id' => 'id']);
    }

    public function setItems($value)
    {
        $this->loadRelated('items', $value);
    }

    /**
     * {@inheritdoc}
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            // Delete all Items related to this model
            LimitChoicesItemTracking::deleteAll(["limit_choices_id" => $this->id]);
            LimitChoicesItem::deleteAll(["limit_choices_id" => $this->id]);
            return true;
        }

        return false;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }

}