<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\limit_choices;

use app\controllers\AjaxController;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\limit_choices\models\LimitChoices;
use app\modules\addons\modules\limit_choices\models\LimitChoicesItem;
use Exception;
use Yii;
use yii\web\Response;
use yii\web\View;

/**
 * Class Module
 * @package app\modules\addons\modules\limit_choices
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "limit_choices";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'LimitChoices' => LimitChoices::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.received' => function ($event) {
                $this->onSubmissionReceived($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
            View::class => [
                'afterRender' => [
                    [Module::class, 'onViewAfterRender']
                ],
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = LimitChoices::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new LimitChoices();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new LimitChoicesItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->limit_choices_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * On Form Before Delete
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = LimitChoices::findAll(['form_id' => $event->sender->id]);
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "app/embed" ) {

            $formModel = $event->sender->context->getFormModel();
            $model = LimitChoices::findOne(['form_id' => $formModel->id, 'status' => LimitChoices::ON]);

            if ($model) {

                $code = "
<script>";
                $fields = [];
                foreach ($model->items as $item) {
                    if (isset($item->field_id)) {
                        // Container
                        if (!in_array($item->field_id, $fields)) {
                            $fields[] = $item->field_id;
                            $code .= <<<JS

var container_{$item->field_id} = $("#{$item->field_id}").closest('div');
JS;
                        }
                        // Field Type
                        $parts = explode('_', $item->field_id);
                        $fieldType = isset($parts[0]) ? $parts[0] : '';
                        // Messages
                        $text = Yii::t('app', '(Available: {choices})', [
                            'choices' => $item->choice_remaining
                        ]);
                        if ($model->remaining_quantity_text) {
                            $text = str_replace('{choices}', $item->choice_remaining, $model->remaining_quantity_text);
                        }
                        if ($item->choice_remaining < 1) {
                            $text = Yii::t('app', '(Not Available)');
                            if ($model->remaining_quantity_error) {
                                $text = str_replace('{choices}', $item->choice_remaining, '(' . $model->remaining_quantity_error . ')');
                            }
                        }
                        if (!$model->remaining_quantity && $item->choice_remaining >= 1)
                            $text = '';
                        // Unavailable
                        $hideUnavailable = $model->unavailable === LimitChoices::UNAVAILABLE_HIDE && $item->choice_remaining === 0 ? 1 : 0;
                        $disableUnavailable = $model->unavailable === LimitChoices::UNAVAILABLE_DISABLE && $item->choice_remaining === 0 ? 1 : 0;
                        $disableFieldUnavailable = $model->unavailable === LimitChoices::UNAVAILABLE_DISABLE_FIELD && $item->choice_remaining === 0 ? 1 : 0;
                        $disableFormUnavailable = $model->unavailable === LimitChoices::UNAVAILABLE_DISABLE_FORM && $item->choice_remaining === 0 ? 1 : 0;
                        // Logic
                        if ($fieldType === 'radio' || $fieldType === 'checkbox') {
                            $code .= <<<CODE

var input_{$item->id} = container_{$item->field_id}.find("input[value='{$item->choice_value}']");
var label_{$item->id} = $("label[for="+input_{$item->id}.attr('id')+"]");
label_{$item->id}.text(label_{$item->id}.text() + ' {$text} ');
// Add validation on radio change and quantity field change
$('input[name="{$item->field_id}"], input[name="{$item->quantity_id}"]').on('change', function() {
    var selectedValue = $('input[name="{$item->field_id}"]:checked').val();
    if (selectedValue != "{$item->choice_value}")
        return;
    
    var choiceRemaining = {$item->choice_remaining};
    var numTravelers = parseInt($('input[name="{$item->quantity_id}"]:checked').val() || 0);
    
    if (numTravelers > choiceRemaining) {
        alert('Not enough spots available for ' + numTravelers + ' travelers. Only ' + choiceRemaining + ' spots remaining.');
        $('input[name="{$item->field_id}"][value="{$item->choice_value}"]').prop('checked', false);
        return false;
    }
});

if ({$hideUnavailable}) {
  input_{$item->id}.closest('div').hide()
} else if ({$disableUnavailable}) {
  input_{$item->id}.attr('disabled', 'disabled');
} else if ({$disableFieldUnavailable}) {
  container_{$item->field_id}.find(':input').attr('disabled', 'disabled');
}
CODE;
                        } elseif ($fieldType === 'selectlist') {
                            $code .= <<<CODE
var option_{$item->id} = container_{$item->field_id}.find("option[value='{$item->choice_value}']");
option_{$item->id}.append(' $text ');

// Add validation on select change
container_{$item->field_id}.find('select').on('change', function() {
    var selectedValue = $(this).val();
    var choiceRemaining = {$item->choice_remaining};
    var numTravelers = parseInt($('input[name="radio_2"]:checked').val() || 0);
    
    if (numTravelers > choiceRemaining) {
        alert('Not enough spots available for ' + numTravelers + ' travelers. Only ' + choiceRemaining + ' spots remaining.');
        $(this).val('');
        return false;
    }
});

if ({$hideUnavailable}) {
  option_{$item->id}.hide()
} else if ({$disableUnavailable}) {
  option_{$item->id}.attr('disabled', 'disabled');
} else if ({$disableFieldUnavailable}) {
  container_{$item->field_id}.find(':input').attr('disabled', 'disabled');
}
CODE;
                        }
                    }
                }
                $code .= "</script>
</body>";

                $content = $event->output;
                $event->output =  str_replace("</body>", $code, $content);
            }
        }
    }

    /**
     * Event Handler
     * Limit Choices when a Submission is received
     *
     * @param $event
     * @throws Exception
     */
    public function onSubmissionReceived($event)
    {

        if (isset($event, $event->form, $event->form->id, $event->submission)) {

            /** @var LimitChoices $model */
            $model = LimitChoices::findOne(['form_id' => $event->form->id, 'status' => LimitChoices::ON]);

            if (isset($model, $model->form_id)) {

                /** @var FormSubmission $submissionModel */
                $submissionModel = $event->submission;
                $submissionData = $submissionModel->data;
                $oldSubmissionData = null;
                if (isset($submissionModel->oldAttributes['data'])) {
                    $oldSubmissionData = is_string($submissionModel->oldAttributes['data'])
                        ? json_decode($submissionModel->oldAttributes['data'], true)
                        : $submissionModel->oldAttributes['data'];
                }

                foreach ($model->items as $item) {
                    if (isset($item->field_id, $submissionData[$item->field_id])) {
                        $value = $submissionData[$item->field_id];

                        // Skip validation when;
                        // 1. We are editing an entry
                        // 2. We haven't changed the selected choices
                        if (!$model->isNewRecord && isset($oldSubmissionData[$item->field_id]) && ($value === $oldSubmissionData[$item->field_id])) {
                            continue;
                        }

                        if (!empty($item->quantity_id) && !empty($submissionData[$item->quantity_id])) {
                            $quantityValue = $submissionData[$item->quantity_id];

                            $this->validate($value, $item, intval($quantityValue));
                        }
                        else
                            $this->validate($value, $item);
                    }
                }
            }
        }
    }

    /**
     * @param string $value
     * @param LimitChoicesItem $item
     */
    protected function validate($value, $item, $quantity = 1)
    {
        if (is_array($value)) {
            // For Checkboxes and Select List
            foreach ($value as $val) {
                $this->validateValue($val, $item, $quantity);
            }
        } else {
            // For Radio Button
            $this->validateValue($value, $item, $quantity);
        }
    }

    /**
     * @param string $value
     * @param LimitChoicesItem $item
     */
    protected function validateValue($value, $item, $quantity = 1)
    {
        if ($item->scope === LimitChoicesItem::SCOPE_CHOICE) {
            if ($value === $item->choice_value) { // Check if value is equal to the expected value
                if ($item->choice_remaining < 1) {
                    $this->showErrorMessage($item);
                } else {
                    $this->updateLimit($item, $quantity);
                }
            }
        } else if ($item->scope === LimitChoicesItem::SCOPE_FIELD) {
            if (is_numeric($value)) { // Check if value is a numeric value
                if ($item->choice_remaining < $value) {
                    $this->showErrorMessage($item);
                } else {
                    $this->updateLimit($item, $value);
                }
            }
        }
    }

    /**
     * @param LimitChoicesItem $item
     * @param null|int $value
     */
    protected function updateLimit($item, $value = null)
    {
        if ($value === null) {
            ++$item->choice_used;
            --$item->choice_remaining;
        } else if (is_numeric($value) && $value > 0) {
            $item->choice_used = $item->choice_used + $value;
            $item->choice_remaining = $item->choice_remaining - $value;
        }
        $item->save();

        // Disable Form
        if ($item->limitChoices->unavailable === LimitChoices::UNAVAILABLE_DISABLE_FORM && $item->choice_remaining < 1) {
            // If it has a custom message
            if (trim($item->limitChoices->remaining_quantity_error) !== "") {
                $item->limitChoices->form->message = $item->limitChoices->remaining_quantity_error;
            }
            $item->limitChoices->form->status = Form::OFF;
            $item->limitChoices->form->save();
        }
    }

    /**
     * @param LimitChoicesItem $item
     */
    protected function showErrorMessage($item)
    {
        $message = !empty($item->choice_message) ? $item->choice_message : Yii::t('app', 'The selected item is no longer available.');
        $errors = [
            'field' => $item->field_id,
            'messages' => [$message],
        ];

        $response = Yii::$app->getResponse();
        $response->format = Response::FORMAT_JSON;
        $response->data = array(
            'action'  => 'submit',
            'success' => false,
            'id' => 0,
            'message' => Yii::t('app', 'There is {startTag}an error in your submission{endTag}.', [
                'startTag' => '<strong>',
                'endTag' => '</strong>',
            ]),
            'errors' => [$errors],
        );
        $response->send();
        exit;
    }
}