<?php

namespace app\modules\addons\modules\insightly\controllers;

use app\components\User;
use app\modules\addons\modules\insightly\helpers\Crm;
use Yii;
use Exception;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\Json;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use app\models\Form;
use app\helpers\ArrayHelper;
use app\modules\addons\modules\insightly\models\Insightly;
use app\modules\addons\modules\insightly\models\InsightlyField;
use app\modules\addons\modules\insightly\models\InsightlySearch;
use app\modules\addons\modules\insightly\services\InsightlyService;

/**
 * AdminController implements the CRUD actions for Insightly model.
 */
class AdminController extends Controller
{
    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'list-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('Insightly')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = Insightly::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\insightly\models\Insightly',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'list-fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($list) {
                    $session = Yii::$app->session;
                    $apiKey = $session['insightly_api_key'];
                    $service = new InsightlyService($apiKey);
                    $fields = Crm::fieldsForList($list);
                    $customFields = $service->getCustomFields($list);
                    return array_merge($fields, $customFields);
                },
                'selectedCallback' => function ($listID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all Insightly models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new InsightlySearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Insightly model.
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new Insightly model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $lists = Crm::lists();
        $connected = false;
        $leadSources = [];
        $leadStatuses = [];

        $request = Yii::$app->request;
        $session = Yii::$app->session;

        $model = new Insightly();
        $fieldModel = new InsightlyField();

        try {
            if ($request->isPost && $model->load($request->post())) {

                $items = Yii::$app->request->post('InsightlyField',[]);

                $service = new InsightlyService($model->api_key);
                $connected = $service->checkConnection();

                if ($connected) {

                    $session['insightly_api_key'] = $model->api_key;
                    $leadStatuses = $service->getLeadStatuses();
                    $leadSources = $service->getLeadSources();

                    if (!empty($items)) {
                        if ($model->validate()) {
                            $model->items = $items;
                            $model->save(false);
                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );
                            return $this->redirect(['index']);
                        } else {
                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify you configuration.')
                            );
                        }
                    }
                } else {
                    // Show error message
                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'Please enter a valid api key.')
                    );
                }
            }
        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
            'lists' => $lists,
            'connected' => $connected,
            'leadSources' => $leadSources,
            'leadStatuses' => $leadStatuses,
        ]);
    }

    /**
     * Updates an existing Insightly model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param $id
     * @return string|\yii\web\Response
     * @throws Exception
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $fieldModel = new InsightlyField();
        $request = Yii::$app->request;
        $session = Yii::$app->session;
        $service = new InsightlyService($model->api_key);
        $connected = $service->checkConnection();
        $session['insightly_api_key'] = $model->api_key;
        $leadSources = [];
        $leadStatuses = [];
        $lists = Crm::lists();

        try {

            if ($request->isPost && $model->load($request->post())) {

                $items = Yii::$app->request->post('InsightlyField',[]);

                if ($connected) {
                    $session['insightly_api_key'] = $model->api_key;
                    if (!empty($items)) {
                        if ($model->validate()) {
                            $model->items = $items;
                            $model->save(false);
                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );
                            return $this->redirect(['index']);
                        } else {
                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify you configuration.')
                            );
                        }
                    }

                } else {
                    // Show error message
                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'Please enter a valid api key.')
                    );
                }
            }

            $leadStatuses = $service->getLeadStatuses();
            $leadSources = $service->getLeadSources();

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
            'lists' => $lists,
            'connected' => $connected,
            'leadSources' => $leadSources,
            'leadStatuses' => $leadStatuses,
        ]);
    }

    /**
     * Enable / Disable multiple CleverReach models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = Insightly::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing Insightly model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the Insightly model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Insightly the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Insightly::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
