<?php

namespace app\modules\addons\modules\import_submissions\controllers;

use app\components\rules\RuleEngine;
use app\components\User;
use app\controllers\AppController;
use app\events\SubmissionEvent;
use app\helpers\ArrayHelper;
use app\helpers\Html;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\modules\import_submissions\models\ImportSubmissions;
use app\modules\addons\modules\import_submissions\models\ImportSubmissionsField;
use app\modules\addons\modules\import_submissions\models\ImportSubmissionsSearch;
use Exception;
use SimpleExcel\SimpleExcel;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\web\Response;
use yii\web\UploadedFile;

/**
 * AdminController implements the CRUD actions for ImportSubmissions model.
 */
class AdminController extends Controller
{

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('ImportSubmissions')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'import', 'process', 'download', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = ImportSubmissions::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\import_submissions\models\ImportSubmissions',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
        ];
    }

    /**
     * Lists all ImportSubmissions models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new ImportSubmissionsSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single ImportSubmissions model.
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new ImportSubmissions model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $request = Yii::$app->request;

        $model = new ImportSubmissions();
        $fieldModel = new ImportSubmissionsField();

        try {

            if ($request->isPost && $model->load($request->post())) {

                $items = Yii::$app->request->post('ImportSubmissionsField',[]);

                if (!empty($items)) {

                    if ($model->validate()) {

                        $model->items = $items;

                        $file = UploadedFile::getInstanceByName('ImportSubmissions[file]');
                        if (isset($file->tempName)) {
                            $fp = fopen($file->tempName, 'r');
                            $str = fread($fp, filesize($file->tempName));
                            fclose($fp);
                            $model->data = $str;
                        }

                        $model->save(false);

                        return $this->redirect(['import', 'id' => $model->id]);

                    } else {
                        // Show error message
                        Yii::$app->getSession()->setFlash(
                            'danger',
                            Yii::t('app', 'Invalid settings found. Please verify you configuration.')
                        );
                    }
                }
            }

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
        ]);
    }

    /**
     * Progress UI to Import Submissions to Form
     *
     * @param $id
     *
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionImport($id)
    {
        $model = $this->findModel($id);

        return $this->render('import', [
            'model' => $model,
        ]);
    }

    /**
     * Submit CSV data to the yMKT API using a batch process
     * @param $id
     * @param int $step
     * @param int $limit
     * @return array
     */
    public function actionProcess($id, $step = 0, $limit = 10)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        try {

            $model = $this->findModel($id);

            if ($model->status === ImportSubmissions::ON) {

                $excel = new SimpleExcel('CSV');

                // Parse CSV data to array
                $excel->parser->loadString($model->data);
                $allRows = $excel->parser->getField();

                // Remove header
                $header = array_shift($allRows);

                // Set number of rows
                $nbRows = count($allRows);

                // Set offset
                $offset = $step * $limit;

                // Batch percentage
                $percentage = round(($offset + $limit) * 100 / $nbRows);

                // All rows have been processed ?
                $completed = ($offset + $limit) >= $nbRows;

                // Reserved column names
                $systemFields = [
                    'created_by',
                    'updated_by',
                    'created_at',
                    'updated_at',
                ];

                $fields = [];

                foreach ($model->items as $item) {
                    array_push($fields, [
                        'form_field' => $item->form_field,
                        'file_field' => $item->file_field,
                    ]);
                }

                $rows = array_slice($allRows, $offset, $limit, false);

                // Batch
                foreach ($rows as $row) {

                    /*******************************
                    /* Prepare data
                    /*******************************/
                    $submissionData = [];

                    foreach ($fields as $field) {
                        if (isset($row[$field['file_field']]) && !in_array($field['form_field'], $systemFields)) {
                            $parts = explode('_', $field['form_field']);
                            $componentType = $parts[0];
                            if (in_array($componentType, ['checkbox', 'selectlist'])) {
                                $values = explode(',', $row[$field['file_field']]);
                                foreach ($values as $value) {
                                    $submissionData[$field['form_field']][] = trim($value);
                                }
                            } else {
                                $submissionData[$field['form_field']] = $row[$field['file_field']];
                            }
                        }
                    }

                    $systemData = [];

                    foreach ($systemFields as $field) {
                        if (isset($row[$field])) {
                            $systemData[$field] = $row[$field];
                        }
                    }

                    /*******************************
                    /* Conditions
                    /*******************************/

                    // By default, import submission
                    $isValid = true;

                    // Conditional Logic
                    if (!empty($model->conditions)) {
                        $engine = new RuleEngine([
                            'conditions' => $model->conditions,
                            'actions' => [],
                        ]);
                        $isValid = $engine->matches($submissionData);
                    }

                    // If it's not valid, continue with the next row
                    if (!$isValid) {
                        continue;
                    }

                    // Set public scenario of the submission
                    $formSubmissionModel = new FormSubmission(['scenario' => 'administration']);

                    // Prepare Submission for validation
                    $postFormSubmission = [
                        'FormSubmission' => [
                            'form_id' => $model->form->id, // Form Model id
                            'data' => $submissionData, // (array)
                        ]
                    ];

                    /*******************************
                    /* Save
                    /*******************************/

                    if ($formSubmissionModel->load($postFormSubmission)) {
                        // Detach behaviors
                        $formSubmissionModel->detachBehaviors();
                        // Save data
                        $formSubmissionModel->created_by = !empty($systemData['created_by']) ? (int) $systemData['created_by'] : Yii::$app->user->id;
                        $formSubmissionModel->updated_by = !empty($systemData['updated_by']) ? (int) $systemData['updated_by'] : Yii::$app->user->id;
                        $formSubmissionModel->created_at = !empty($systemData['created_at']) ? strtotime($systemData['created_at']) ? strtotime($systemData['created_at']) : time() : time();
                        $formSubmissionModel->updated_at = !empty($systemData['updated_at']) ? strtotime($systemData['updated_at']) ? strtotime($systemData['updated_at']) : time() : time();;
                        $saved = $formSubmissionModel->save((bool) $model->run_validation);

                        if ($saved) {
                            // Trigger event
                            $events = is_array($model->events) ? $model->events : explode(',', $model->events);
                            if (in_array(AppController::EVENT_SUBMISSION_ACCEPTED, $events)) {
                                Yii::$app->trigger(AppController::EVENT_SUBMISSION_ACCEPTED, new SubmissionEvent([
                                    'sender' => $this,
                                    'form' => $model->form,
                                    'submission' => $formSubmissionModel,
                                ]));
                            }
                        } else {
                            // Save failed records
                            if (empty($model->invalid_data)) {
                                // Add header
                                $excel->writer->addRow(array_merge($header, ['Error Messages']), false);
                            }
                            // Add row
                            $excel->writer->addRow(array_merge($row, [json_encode($formSubmissionModel->getErrors())]), true);
                            $model->invalid_data = $excel->writer->saveString();
                            $model->save();
                        }

                    }

                }

                $message = Yii::t('app',
                    'We have processed a batch of {0} records.', [
                        Html::tag('span', count($rows), ['style' => 'font-weight: bold']),
                    ]);

                if ($completed) {
                    $model->status = ImportSubmissions::OFF;
                    $model->save();

                    $message = Yii::t('app',
                        'We have processed {0} records: {1} imported.', [
                            Html::tag('span', $nbRows, ['style' => 'font-weight: bold']),
                            Html::tag('span', $nbRows, ['style' => 'font-weight: bold']),
                        ]);

                    if (!empty($model->invalid_data)) {
                        $excel->parser->loadString($model->invalid_data);
                        $allInvalidRows = $excel->parser->getField();
                        $rejected = count($allInvalidRows) - 1;
                        $imported = $nbRows - $rejected;
                        if ($rejected > 0) {
                            $imported = $imported < 0 ? 0 : $imported;
                            $message = Yii::t('app',
                                'We have processed {0} records: {1} imported and {2} rejected due to a validation error ({3}).', [
                                    Html::tag('span', $nbRows, ['style' => 'font-weight: bold']),
                                    Html::tag('span', $imported, ['style' => 'font-weight: bold']),
                                    Html::tag('span', $rejected, ['style' => 'font-weight: bold', 'class' => 'text-danger']),
                                    Html::a(Yii::t('app', 'Download failed records'), ['download', 'id' => $model->id]),
                                ]);
                        }
                    }
                }

                return [
                    'completed' => $completed,
                    'step' => $completed ? 'done' : $step + 1,
                    'percentage' => $completed ? 100 : $percentage,
                    'limit' => $limit,
                    'message' => $message,
                ];

            }

        } catch (Exception $e) {
            Yii::error($e);
            return [
                'error' => 1,
            ];
        }

        return [
            'error' => 2,
        ];
    }

    public function actionDownload($id)
    {
        $model = $this->findModel($id);

        return Yii::$app->response->sendContentAsFile($model->invalid_data, sprintf("%s.%s", 'Failed_Records', 'csv'), [
            'mimeType' => 'application/csv',
            'inline'   => false
        ]);
    }

    /**
     * Enable / Disable multiple CleverReach models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = ImportSubmissions::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing ImportSubmissions model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the ImportSubmissions model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return ImportSubmissions the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = ImportSubmissions::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
