<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\hubspot_forms;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\ArrayHelper;
use app\helpers\SlugHelper;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\hubspot_forms\models\HubspotForm;
use app\modules\addons\modules\hubspot_forms\models\HubspotFormItem;
use Exception;
use Yii;
use yii\httpclient\Client;

class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "hubspot_forms";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'HubspotForm' => HubspotForm::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            'app\models\Form' => [
                'beforeDelete' => [
                    ['app\modules\addons\modules\hubspot_forms\Module', 'beforeDeleteAForm']
                ]
            ],
            'yii\base\View' => [
                'afterRender' => [
                    ['app\modules\addons\modules\hubspot_forms\Module', 'afterRenderView']
                ],
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = HubspotForm::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new HubspotForm();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new HubspotFormItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->hubspot_forms_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a Form model is deleted
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function beforeDeleteAForm($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = HubspotForm::findAll(['form_id' => $event->sender->id]);
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function afterRenderView($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "app/embed" ) {

            $formModel = $event->sender->context->getFormModel();
            $model = HubspotForm::findOne(['form_id' => $formModel->id, 'status' => HubspotForm::ON]);

            if ($model && !empty($model->tracking_code)) {

                // Add client tracking code to app/embed view
                $code = <<<CODE

{$model->tracking_code}
</body>
CODE;
                $content = $event->output;
                $event->output =  str_replace("</body>", $code, $content);
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }
        /*******************************
        /* Send Hubspot Forms
        /*******************************/
        $this->sendSubmissionData($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Send Hubspot Forms
        /*******************************/
        $this->sendSubmissionData($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Send the form submission data through an HTTP POST request
     * either as URL encoded form data or as a JSON string
     * depending on the format selected in the HubspotForm configuration
     *
     * @param Form $formModel
     * @param FormSubmission $submissionModel
     * @param array $filePaths
     * @param $event
     */
    public function sendSubmissionData($formModel, $submissionModel, $filePaths, $event)
    {
        try {

            $models = HubspotForm::findAll(['form_id' => $formModel->id, 'status' => 1]);
            $dataModel = $formModel->formData;
            // Submission data for rule engine
            $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

            /*******************************
            /* Process Hubspot Forms
            /*******************************/
            foreach ($models as $model) {
                // Only when the required event occurs
                if ($model->event !== $event) {
                    continue;
                }

                // By default, valid if there are no conditions
                $isValid = empty($model->conditions);

                // Conditional Logic
                if (!$isValid) {
                    $engine = new RuleEngine([
                        'conditions' => $model->conditions,
                        'actions' => [],
                    ]);
                    $isValid = $engine->matches($data);
                }

                // If the conditions have been met
                if ($isValid) {

                    $client = new Client([
                        'transport' => 'yii\httpclient\CurlTransport'
                    ]);

                    $submissionData = $submissionModel->getSubmissionData();

                    $fields = [];

                    foreach ($model->items as $item) {
                        $value = null;
                        if (isset($submissionData[$item->field_id])) {
                            $value = $submissionData[$item->field_id];
                            if (is_array($value)) {
                                $value = implode(';', $value);
                            }
                        }
                        $field = [
                            'name' => $item->field_name,
                            'value' => $value,
                        ];
                        array_push($fields, $field);
                    }

                    // Header
                    $headers = ['User-Agent' => Yii::$app->name];
                    // Format
                    $headers['content-type'] = 'application/json';
                    $format = Client::FORMAT_JSON;

                    // Body
                    $body = [
                        'submittedAt' => (int) $submissionModel->created_at * 1000, // In milliseconds
                        'fields' => $fields,
                    ];

                    if ($model->context) {
                        $body['context'] = [
                            'hutk' => !empty($_COOKIE['hubspotutk']) ? $_COOKIE['hubspotutk'] : null,
                            'pageUri' => !empty($submissionData['url']) ? $submissionData['url'] : null,
                            'ipAddress' => !empty($submissionData['ip_address']) ? $submissionData['ip_address'] : null,
                        ];
                    }

                    if (empty($model->consent)) {
                        $body['consent'] = $model->consent;
                    }

                    // Send HTTP POST request
                    $response = $client->createRequest()
                        ->setMethod('POST')
                        ->setFormat($format)
                        ->setUrl($model->url)
                        ->addHeaders($headers)
                        ->setData($body)
                        ->setOptions([
                            CURLOPT_SSL_VERIFYHOST => 0,
                            CURLOPT_SSL_VERIFYPEER => 0
                        ])
                        ->send();

                    // Log error response
                    $response = $response->getData();
                    if (is_array($response) && isset($response['status']) && $response['status'] === 'error') {
                        Yii::error($response);
                    }
                }
            }

        } catch (Exception $e) {
            // Log error
            Yii::error($e);
        }
    }
}
