<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\google_sheets\services;

use GuzzleHttp\Exception\GuzzleException;
use Yii;
use yii\helpers\Url;

/**
 * Class GoogleSheetsService
 * @package app\modules\addons\modules\google_sheets\services
 */
class GoogleSheetsService
{
    public $client;

    /** @var \Google_Service_Sheets */
    public $service;
    public $config;

    private $clientID;
    private $clientSecret;

    /**
     * GoogleSheetsService constructor.
     * @throws \Google_Exception
     * @throws \Exception
     */
    public function __construct()
    {
        require_once(Yii::getAlias('@addons/google_sheets/services/vendor/autoload.php'));

        $this->clientID = Yii::$app->settings->get('addon_google_sheets.clientId');
        $this->clientSecret = Yii::$app->settings->get('addon_google_sheets.clientSecret');

        $this->config = [
            'web' => [
                'client_id' => $this->clientID,
                'client_secret' => $this->clientSecret,
                'redirect_uris' => [
                    Url::to(['/addons/google_sheets/admin/authorized'], true)
                ]
            ]
        ];

        $this->client = new \Google_Client();
        $this->client->setAuthConfig($this->config);
        $this->client->addScope(\Google_Service_Sheets::SPREADSHEETS);
        $this->client->addScope(\Google_Service_Sheets::DRIVE_READONLY);
        $this->client->setAccessType('offline');
        $this->client->setPrompt('consent');
        $this->client->setState(bin2hex(random_bytes(128/8)));
        $this->client->setIncludeGrantedScopes(true);
    }

    /**
     * Set redirect url
     *
     * @param $uri
     */
    public function setRedirectUri($uri)
    {
        $this->client->setRedirectUri($uri);
    }

    /**
     * Redirect to Authentication URL
     */
    public function redirectToAuthUrl()
    {
        $auth_url = $this->client->createAuthUrl();
        header('Location: ' . filter_var($auth_url, FILTER_SANITIZE_URL));
        exit;
    }

    /**
     * Fetch Access Token With Auth Code
     *
     * @param $code
     * @return array
     */
    public function fetchAccessTokenWithAuthCode($code): array
    {
        return $this->client->fetchAccessTokenWithAuthCode($code);
    }

    /**
     * Set Access Token
     *
     * @param $token
     */
    public function setAccessToken($token)
    {
        $this->client->setAccessToken($token);

        if ($this->client->isAccessTokenExpired()) {
            if ($this->client->getRefreshToken()) {
                $this->client->fetchAccessTokenWithRefreshToken($this->client->getRefreshToken());
            }
        }
    }

    /**
     * Get Access Token
     *
     * @return array
     */
    public function getAccessToken(): array
    {
        return $this->client->getAccessToken();
    }

    /**
     * Get Google Sheets Service
     *
     * @return \Google_Service_Sheets
     */
    public function getService()
    {
        if (empty($this->service)) {
            $this->service = new \Google_Service_Sheets($this->client);
        }

        return $this->service;
    }

    /**
     * Get Spreadsheet list
     *
     * https://developers.google.com/drive/api/v3/reference/files/list
     *
     * @return false|mixed
     * @throws GuzzleException
     */
    public function getSpreadsheetList()
    {
        $httpClient = $this->client->authorize();
        $token = $this->getAccessToken();
        $endpoint = "https://www.googleapis.com/drive/v3/files?q=mimeType%20%3D%20'application%2Fvnd.google-apps.spreadsheet'&pageSize=1000&access_token={$token['access_token']}";
        $response = $httpClient->get($endpoint);

        $responseBody = $response->getBody();
        if (empty($responseBody)) {
            return false;
        }

        $body = json_decode($responseBody, true);
        $list = [];
        foreach ($body['files'] as $file) {
            $list[] = [
                'id' => $file['id'],
                'name' => $file['name'],
            ];
        }

        return $list;
    }

    /**
     * Get Sheet List
     *
     * @param $spreadsheetID
     * @return array|false
     * @throws GuzzleException
     */
    public function getSheetList($spreadsheetID)
    {
        $httpClient = $this->client->authorize();
        $token = $this->getAccessToken();
        $endpoint = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheetID}/?access_token={$token['access_token']}";
        $response = $httpClient->get($endpoint);

        $responseBody = $response->getBody();
        if (empty($responseBody)) {
            return false;
        }

        $body = json_decode($responseBody, true);
        $list = [];
        foreach ($body['sheets'] as $sheet) {
            $list[] = [
                'id' => $sheet['properties']['title'],
                'name' => $sheet['properties']['title'],
            ];
        }

        return $list;
    }

    /**
     * Get Column List
     *
     * @param $spreadsheetID
     * @param $sheetName
     * @return array|false
     * @throws GuzzleException
     */
    public function getColumnList($spreadsheetID, $sheetName)
    {
        $httpClient = $this->client->authorize();
        $token = $this->getAccessToken();
        $sheetName = urlencode($sheetName);
        $endpoint = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheetID}/values/{$sheetName}!A1:ZZ1?access_token={$token['access_token']}";
        $response = $httpClient->get($endpoint);

        $responseBody = $response->getBody();
        if (empty($responseBody)) {
            return false;
        }

        $body = json_decode($responseBody, true);

        if ( !isset( $body['values'][0] ) ) {
            return false;
        }

        $keys = [];
        for ($x = "A";; $x++) {
            $keys[] = $x;
            if ($x === "ZZ") { break; }
        }

        $columns = array_combine(array_slice($keys, 0, count($body['values'][0])), $body['values'][0]);

        $list = [];
        foreach ($columns as $column => $title) {
            $list[] = [
                'id' => $column,
                'name' => $title,
            ];
        }

        return $list;
    }

    /**
     * Append New Row
     *
     * @param $spreadsheetID
     * @param $sheetName
     * @param $data
     * @return false|array
     * @throws GuzzleException
     */
    public function appendNewRow($spreadsheetID, $sheetName, $data)
    {
        if (empty($sheetName) || empty($data) ) {
            return false;
        }

        $columnData = [];
        foreach ($data as $letter => $value) {
            $column = $this->letterToColumn($letter);
            $columnData[$column] = is_array($value) ? implode(',', $value) : $value;
        }
        $lastColumn = max(array_keys($columnData));
        $lastLetter = $this->columnToLetter($lastColumn);

        $values = array();
        for ($c = 1; $c <= $lastColumn; $c++) {
            $value = '';
            foreach ($columnData as $column => $v) {
                if ($c === $column) {
                    $value = is_array($v) ? implode(',', $v) : $v;
                }
            }
            $values[] = $value;
        }

        $httpClient = $this->client->authorize();
        $token = $this->getAccessToken();
        $sheetName = urlencode($sheetName);
        $url = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheetID}/values/{$sheetName}!A:{$lastLetter}:append?valueInputOption=USER_ENTERED";
        $sheetName = urldecode($sheetName);
        $options = [
            'headers' => [
                'Authorization' => 'Bearer ' . $token['access_token'],
                'Content-Type'  => 'application/json',
            ],
            'body'    => '{"range":"' . $sheetName . '!A:' . $lastLetter . '","majorDimension":"ROWS","values":[' . json_encode($values) . ']}',
        ];

        $response = $httpClient->post($url, $options);
        $responseBody = $response->getBody();

        if (empty($responseBody)) {
            Yii::error('Empty response.', 'google_sheets');
            return false;
        }

        $body = json_decode($responseBody, true);

        if (isset($body['error'])) {
            Yii::error($body['error'], 'google_sheets');
            return false;
        }

        return $body;
    }

    /**
     * Get column from letter
     *
     * @param $l
     * @return float|int
     */
    public function letterToColumn($l) {
        $num = 0;
        $arr = array_reverse(str_split($l));

        for ($i = 0; $i < count($arr); $i++) {
            $num += (ord(strtolower($arr[$i])) - 96) * (pow(26,$i));
        }
        return $num;
    }

    /**
     * Get letter from column
     *
     * @param $c
     * @return string
     */
    public function columnToLetter($c): string
    {

        $c = intval($c);
        if ($c <= 0) return '';

        $letter = '';

        while($c != 0){
            $p = ($c - 1) % 26;
            $c = intval(($c - $p) / 26);
            $letter = chr(65 + $p) . $letter;
        }

        return $letter;

    }

}