<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\google_sheets\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\modules\addons\models\Addon;
use app\modules\addons\modules\google_sheets\models\GoogleSheets;
use app\modules\addons\modules\google_sheets\models\GoogleSheetsItem;
use app\modules\addons\modules\google_sheets\models\GoogleSheetsSearch;
use app\modules\addons\modules\google_sheets\services\GoogleSheetsService;
use kartik\depdrop\DepDropAction;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\Html;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\Controller;
use yii\web\NotFoundHttpException;

/**
 * AdminController implements the CRUD actions for Ga model.
 */
class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'sheets', 'columns'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['settings'], 'allow' => true, 'roles' => ['configureAddons'], 'roleParams' => function() {
                        return ['model' => Addon::findOne(['id' => 'google_sheets'])];
                    }],
                    ['actions' => ['create', 'authorize', 'authorized'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } elseif ($postData = Yii::$app->request->post('GoogleSheets')) {
                            if (!empty($postData['form_id']) ) {
                                return ['model' => Form::findOne(['id' => $postData['form_id']])];
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = GoogleSheets::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\google_sheets\models\GoogleSheets',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                $fields = SubmissionHelper::getFieldsForFieldMapping($fields);
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'sheets' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($spreadsheetID, $params) {
                    $output = array();
                    $token = null;
                    $formID = null;
                    if (isset($_POST['depdrop_all_params']['googlesheets-form_id'])) {
                        $formID = $_POST['depdrop_all_params']['googlesheets-form_id'];
                    }
                    $model = GoogleSheets::findOne(['form_id' => $formID]);

                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                            $token = !is_string($model->oauth) ? $model->oauth : Json::decode($model->oauth);
                        }
                    } elseif (Yii::$app->session->has('google_access_token')) {
                        $token = Yii::$app->session->get('google_access_token');
                    }

                    if ($token && $spreadsheetID) {
                        $service = new GoogleSheetsService();
                        $service->setAccessToken($token);
                        $output = $service->getSheetList($spreadsheetID);
                    }

                    return $output;
                },
                'selectedCallback' => function ($id, $params) {
                    if (isset($_POST['depdrop_params'])) {
                        $ids = $_POST['depdrop_params'];
                        return empty($ids[0]) ? null : $ids[0];
                    }
                }
            ],
            'columns' => [
                'class' => DepDropAction::class,
                'outputCallback' => function () {
                    $output = array();

                    if (isset($_POST['depdrop_parents'])) {
                        $ids = $_POST['depdrop_parents'];
                        $spreadsheetID = empty($ids[0]) ? null : $ids[0];
                        $sheetName = empty($ids[1]) ? null : $ids[1];
                        $token = null;
                        $formID = null;
                        if (isset($_POST['depdrop_all_params']['googlesheets-form_id'])) {
                            $formID = $_POST['depdrop_all_params']['googlesheets-form_id'];
                        }
                        $model = GoogleSheets::findOne(['form_id' => $formID]);
                        if (!empty($model->form)) {
                            if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                                $token = !is_string($model->oauth) ? $model->oauth : Json::decode($model->oauth);
                            }
                        } elseif (Yii::$app->session->has('google_access_token')) {
                            $token = Yii::$app->session->get('google_access_token');
                        }

                        if ($token && $spreadsheetID && $sheetName) {
                            $service = new GoogleSheetsService();
                            $service->setAccessToken($token);
                            return $service->getColumnList($spreadsheetID, $sheetName);
                        }
                    }

                    return $output;
                },
                'selectedCallback' => function ($id, $params) {
                    if (isset($_POST['depdrop_params'])) {
                        $ids = $_POST['depdrop_params'];
                        return empty($ids[0]) ? null : $ids[0];
                    }
                }
            ],
        ];
    }

    /**
     * Lists all Google Sheets models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new GoogleSheetsSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single GoogleSheets model.
     *
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new GoogleSheets model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $model = new GoogleSheets();
        $itemModel = new GoogleSheetsItem();
        $token = Yii::$app->session->get('google_access_token');
        $spreadsheets = [];

        try {
            if ($token) {

                // Test connection to verify that the access token is valid
                $service = new GoogleSheetsService();
                $service->setAccessToken($token);
                $spreadsheets = $service->getSpreadsheetList();
                $spreadsheets = ArrayHelper::map($spreadsheets, 'id', 'name');

                if ($model->load(Yii::$app->request->post())) {
                    $items = Yii::$app->request->post('GoogleSheetsItem',[]);
                    if (!empty($items)) {
                        $model->oauth = !is_string($token) ? Json::encode($token) : $token;
                        if ($model->validate()) {
                            $model->items = $items;
                            $model->save(false);
                            Yii::$app->getSession()->setFlash(
                                'success',
                                Yii::t('app', 'The form has been successfully configured.')
                            );
                            return $this->redirect(['index']);
                        } else {
                            // Show error message
                            Yii::$app->getSession()->setFlash(
                                'danger',
                                Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                            );
                        }
                    }
                }
            }

        } catch (\Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'itemModel' => $itemModel,
            'forms' => $forms,
            'spreadsheets' => $spreadsheets,
            'token' => $token,
        ]);

    }

    /**
     * Authorize Google Account
     *
     * @param null $id
     */
    public function actionAuthorize($id = null)
    {
        $url = Url::to(['/addons/google_sheets/admin/create'], true);

        if ($id) {
            $url = Url::to(['/addons/google_sheets/admin/update', 'id' => $id], true);
        }

        Yii::$app->session->set('redirect_url', $url);

        $service = new GoogleSheetsService();
        $service->setRedirectUri(Url::to(['/addons/google_sheets/admin/authorized'], true));
        $service->redirectToAuthUrl();
    }

    /**
     * Get Access Token form Authorized Google Account
     *
     * @return \yii\web\Response
     */
    public function actionAuthorized()
    {
        $error = Yii::$app->request->get('error'); // E.g. 'access_denied'
        $state = Yii::$app->request->get('state');
        $code = Yii::$app->request->get('code'); // Authorization code
        $url = Yii::$app->session->get('redirect_url');

        if (empty($error) && !empty($code)) {
            $service = new GoogleSheetsService();
            $token = $service->fetchAccessTokenWithAuthCode($code);
            // Save token in user session
            Yii::$app->session->set('google_access_token', $token);

            if (empty($url)) {
                Yii::$app->getSession()->setFlash(
                    'success',
                    Yii::t('app', 'Your access token has been successfully configured.')
                );
                $url = ['index'];
            }

            return $this->redirect($url);
        }
    }

    /**
     * Updates an existing GoogleSheets model.
     * If update is successful, the browser will be redirected to the 'view' page.
     *
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $spreadsheets = null;
        $token = !empty($model->oauth) && is_string($model->oauth) ? Json::decode($model->oauth) : $model->oauth;

        if (Yii::$app->session->has('google_access_token')) {
            $token = Yii::$app->session->get('google_access_token');
            $model->oauth = Json::encode($token);
            $model->save();
        }

        try {
            // Test connection to verify that the access token is valid
            $service = new GoogleSheetsService();
            $service->setAccessToken($token);
            $spreadsheets = $service->getSpreadsheetList();
            // Update oauth credentials
            $updatedToken = $service->getAccessToken();
            if (!empty($updatedToken['refresh_token']) && $updatedToken['refresh_token'] !== $token['refresh_token']) {
                $model->oauth = !is_string($updatedToken) ? Json::encode($updatedToken) : $updatedToken;
                $model->save(false);
            }
            $spreadsheets = ArrayHelper::map($spreadsheets, 'id', 'name');

            if ($model->load(Yii::$app->request->post())) {
                $items = Yii::$app->request->post('GoogleSheetsItem',[]);
                if (!empty($items)) {
                    if ($model->validate()) {
                        $model->items = $items;
                        $model->save(false);
                        Yii::$app->getSession()->setFlash(
                            'success',
                            Yii::t('app', 'The form has been successfully configured.')
                        );
                        return $this->redirect(['index']);
                    } else {
                        // Show error message
                        Yii::$app->getSession()->setFlash(
                            'danger',
                            Yii::t('app', 'Invalid settings found. Please verify your configuration.')
                        );
                    }
                }
            }

        } catch (\Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            $refreshLink = ' - ' . Yii::t('app', 'Authenticate your Google Sheets account to update your integration.');
			if (!empty($model->oauth)) {
                $refreshLink = ' - ' . Html::a(Yii::t('app', 'Authenticate your Google Sheets account to update your integration.'), Url::current(), ['class' => 'text-danger']);
            }
            Yii::$app->getSession()->setFlash('danger', $e->getMessage() . ' ' . $refreshLink);
            // Reset access token
            $model->oauth = null;
            $model->save();
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'itemModel' => new GoogleSheetsItem,
            'forms' => $forms,
            'spreadsheets' => $spreadsheets,
            'token' => $token,
        ]);
    }

    /**
     * Global settings for Google Sheets integration
     *
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionSettings()
    {
        $request = Yii::$app->request;
        $settings = Yii::$app->settings;

        if ($request->post()) {
            if ($request->post('action') === 'global-settings') {
                try {

                    $clientID = strip_tags($request->post('addon_google_sheets_clientId', $settings->get('addon_google_sheets.clientId')));
                    $clientSecret = strip_tags($request->post('addon_google_sheets_clientSecret', $settings->get('addon_google_sheets_clientSecret')));
                    $settings->set('addon_google_sheets.clientId', $clientID);
                    $settings->set('addon_google_sheets.clientSecret', $clientSecret);
                    // Show success alert
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'Google Sheets Integration Settings have been successfully updated.')
                    );
                } catch (\Exception $e) {
                    // Clear settings
                    $settings->set('addon_google_sheets.clientId', '');
                    $settings->set('addon_google_sheets.clientSecret', '');
                    // Log
                    Yii::error($e);
                    // Show error message
                    Yii::$app->getSession()->setFlash('danger', $e->getMessage());
                }
            }
        }

        return $this->render('settings');
    }

    /**
     * Enable / Disable multiple Google Analytics GoogleSheets
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function actionUpdateStatus($status)
    {
        $models = GoogleSheets::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        } else {
            foreach ($models as $model) {
                $model->status = $status;
                $model->update();
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The selected items have been successfully updated.')
            );
            return $this->redirect(['index']);
        }
    }

    /**
     * Deletes an existing GoogleSheets model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     *
     * @param $id
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        Yii::$app->getSession()->setFlash('success', Yii::t('app', 'The form tracking has been successfully deleted.'));

        return $this->redirect(['index']);
    }

    /**
     * Finds the GoogleSheets model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return GoogleSheets the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = GoogleSheets::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
