<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\google_places_autocomplete;

use app\controllers\AjaxController;
use app\models\Form;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\google_places_autocomplete\models\GooglePlacesAutoComplete;
use app\modules\addons\modules\google_places_autocomplete\models\GooglePlacesAutoCompleteItem;

/**
 * Class Module
 * @package app\modules\addons\modules\google_places_autocomplete
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

	public $id = "google_places_autocomplete";
	public $defaultRoute = 'admin/index';
	public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'GooglePlacesAutoComplete' => GooglePlacesAutoComplete::class,
        ];
    }

	/**
	 * @inheritdoc
	 */
	public function attachGlobalEvents()
	{
		return [
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
	}

	/**
	 * @inheritdoc
	 */
	public function attachClassEvents()
	{
		return [
			'app\models\Form' => [
				'beforeDelete' => [
					['app\modules\addons\modules\google_places_autocomplete\Module', 'beforeDeleteAForm']
				]
			],
			'yii\base\View' => [
				'afterRender' => [
					['app\modules\addons\modules\google_places_autocomplete\Module', 'setupGooglePlacesAPI']
				],
			],
		];
	}

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = GooglePlacesAutoComplete::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new GooglePlacesAutoComplete();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new GooglePlacesAutoCompleteItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

	/**
	 * Event Handler
	 * Before a Form model is deleted
	 * @param $event
	 *
	 * @throws \Exception
	 */
	public static function beforeDeleteAForm($event)
	{
		if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
			/** @var \app\modules\addons\modules\briteverify\models\GooglePlacesAutoComplete $model */
			$model = GooglePlacesAutoComplete::findOne(['form_id' => $event->sender->id]);
			if ($model) {
				$model->delete();
			}
		}
	}

	/**
	 * Event Handler
	 * After a view is rendered
	 *
	 * @param $event
	 */
	public static function setupGooglePlacesAPI($event)
	{
		if (isset($event, $event->sender, $event->sender->context) &&
		    isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
		    $event->sender->context->module->requestedRoute === "app/embed" ) {

            $formModel = $event->sender->context->getFormModel();
			$gpa = GooglePlacesAutoComplete::findOne(['form_id' => $formModel->id, 'status' => 1]);

			if ($gpa) {

                $initAutocomplete = '';
				$itemCode = '';
				$cssCode = '';

				foreach ($gpa->items as $item) {
                    if (isset($item->field_id)) {
					    // Init JS code
                        $initAutocomplete .= "init_autocomplete_{$item->field_id}(); ";
					    // Get fields
                        $components = [];
                        if (!empty($item->street_number_field)) {
                            array_push($components, [
                                'id' => $item->street_number_field,
                                'type' => 'street_number',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->route_field)) {
                            array_push($components, [
                                'id' => $item->route_field,
                                'type' => 'route',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->sublocality_level_1_field)) {
                            array_push($components, [
                                'id' => $item->sublocality_level_1_field,
                                'type' => 'sublocality_level_1',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->sublocality_level_2_field)) {
                            array_push($components, [
                                'id' => $item->sublocality_level_2_field,
                                'type' => 'sublocality_level_2',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->locality_field)) {
                            array_push($components, [
                                'id' => $item->locality_field,
                                'type' => 'locality',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->administrative_area_level_1_field)) {
                            array_push($components, [
                                'id' => $item->administrative_area_level_1_field,
                                'type' => 'administrative_area_level_1',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->administrative_area_level_2_field)) {
                            array_push($components, [
                                'id' => $item->administrative_area_level_2_field,
                                'type' => 'administrative_area_level_2',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->administrative_area_level_3_field)) {
                            array_push($components, [
                                'id' => $item->administrative_area_level_3_field,
                                'type' => 'administrative_area_level_3',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->country_field)) {
                            array_push($components, [
                                'id' => $item->country_field,
                                'type' => 'country',
                                'size' => 'long_name',
                            ]);
                        }
                        if (!empty($item->country_code_field)) {
                            array_push($components, [
                                'id' => $item->country_code_field,
                                'type' => 'country',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->postal_code_field)) {
                            array_push($components, [
                                'id' => $item->postal_code_field,
                                'type' => 'postal_code',
                                'size' => 'short_name',
                            ]);
                        }
                        if (!empty($item->latitude_field)) {
                            array_push($components, [
                                'id' => $item->latitude_field,
                                'type' => 'lat'
                            ]);
                        }
                        if (!empty($item->longitude_field)) {
                            array_push($components, [
                                'id' => $item->longitude_field,
                                'type' => 'lng'
                            ]);
                        }
                        $component = json_encode($components);
                        // Restrictions
                        $options = [];
                        if (!empty($item->type)) {
                            $options['types'] = [$item->type];
                        }
                        if (!empty($item->country)) {
                            $options['componentRestrictions'] = ['country' => $item->country];
                        }
                        if (!empty($options)) {
                            $options = json_encode($options);
                        } else {
                            $options = '{}';
                        }
                        // Map Selector
                        $selector = !empty($item->map_selector) ? $item->map_selector : '';
                        // Map Zoom
                        $zoom = !empty($item->map_zoom) ? $item->map_zoom : 3;
                        // Show Marker
                        $showMarker = (boolean) !empty($item->map_marker) ? $item->map_marker : 0;
                        // Default CSS Code for Map DIVs
                        if (!empty($selector)) {
                            $cssCode .= "
#{$selector} {
    width: 100%;
    min-height: 300px;
}
";
                        }
						// Set auto complete to each field
						$itemCode .= <<<ITEM_CODE
var autocomplete_{$item->field_id}, marker_{$item->field_id}, map_{$item->field_id};
var component_{$item->field_id} = {$component};
var selector_{$item->field_id} = '{$selector}';
var show_marker_{$item->field_id} = {$showMarker};

function init_autocomplete_{$item->field_id}() {
  autocomplete_{$item->field_id} = new google.maps.places.Autocomplete(
      document.getElementById('{$item->field_id}'), {$options});

  var lat = 39.5;
  var lng = -98.35;
  var zoom = {$zoom};
  var lat_field = document.getElementById('{$item->latitude_field}');
  var lng_field = document.getElementById('{$item->longitude_field}');
  
  if (geolocation) {
    if (window.browserGeolocation) {
      lat = window.browserGeolocation.lat;
      lng = window.browserGeolocation.lng;
    }  
  }

  if (lat_field && lng_field && lat_field.length && lng_field.length && lat_field.value && lng_field.value) {
    lat = parseFloat(lat_field.value);
    lng = parseFloat(lng_field.value);
  }

  var options_{$item->field_id} = {
    zoom: zoom,
    center: { lat: lat, lng: lng }
  };

  if (selector_{$item->field_id}.length > 0) {
      map_{$item->field_id} = new google.maps.Map(
      document.getElementById(selector_{$item->field_id}), options_{$item->field_id});
    
      marker_{$item->field_id} = new google.maps.Marker({
        position: { lat: lat, lng: lng },
        map: map_{$item->field_id},
        draggable: true
      });
      
      if (!show_marker_{$item->field_id}) {
        marker_{$item->field_id}.setVisible(false);
      }
    
      google.maps.event.addListener(marker_{$item->field_id}, 'dragend', function() {
        document.getElementById('{$item->latitude_field}').value = marker_{$item->field_id}.getPosition().lat();
        document.getElementById('{$item->longitude_field}').value = marker_{$item->field_id}.getPosition().lng();
      });
    
      map_{$item->field_id}.addListener('click', function(e) {
        marker_{$item->field_id}.setPosition(e.latLng);
        document.getElementById('{$item->latitude_field}').value = marker_{$item->field_id}.getPosition().lat();
        document.getElementById('{$item->longitude_field}').value = marker_{$item->field_id}.getPosition().lng();
      });
  }

  autocomplete_{$item->field_id}.setFields(['address_component', 'geometry']);
  autocomplete_{$item->field_id}.addListener('place_changed', update_autocomplete_{$item->field_id});
}

function update_autocomplete_{$item->field_id}() {
  var place = autocomplete_{$item->field_id}.getPlace();

  if (!place || !place.geometry) {
    window.alert("No details available for input: '" + place.name + "'");
    return;
  }

  if (selector_{$item->field_id}.length > 0) {
    marker_{$item->field_id}.setVisible(false);
    if (place.geometry.viewport) {
      map_{$item->field_id}.fitBounds(place.geometry.viewport);
    } else {
      map_{$item->field_id}.setCenter(place.geometry.location);
      map_{$item->field_id}.setZoom(17);
    }
    marker_{$item->field_id}.setPosition(place.geometry.location);
    if (show_marker_{$item->field_id}) {
      marker_{$item->field_id}.setVisible(true);
    }
  }

  for (var c = 0; c < component_{$item->field_id}.length; c++) {
    document.getElementById(component_{$item->field_id}[c].id).value = '';    
  }

  for (var i = 0; i < place.address_components.length; i++) {
    var addressType = place.address_components[i].types[0];
    for (var k = 0; k < component_{$item->field_id}.length; k++) {
      if (component_{$item->field_id}[k].type === addressType) {
        var val = place.address_components[i][component_{$item->field_id}[k].size];
        document.getElementById(component_{$item->field_id}[k].id).value = val;
      } else if (component_{$item->field_id}[k].type === 'lat') {
        document.getElementById(component_{$item->field_id}[k].id).value = place.geometry.location.lat();
      } else if (component_{$item->field_id}[k].type === 'lng') {
        document.getElementById(component_{$item->field_id}[k].id).value = place.geometry.location.lng();
      }
    }
  }
}

if (geolocation) {
  document.getElementById('{$item->field_id}').addEventListener("focus", function() {
    if (window.browserGeolocation && window.browserCoordsAccuracy) {
      var circle = new google.maps.Circle({center: window.browserGeolocation, radius: window.browserCoordsAccuracy});
      autocomplete_{$item->field_id}.setBounds(circle.getBounds());
      if (typeof $ !== 'undefined') {
        $("form").trigger("geolocation", [window.browserGeolocation]);
      }
    }
  }, true);
}

ITEM_CODE;
					}
				}

				$jsCode = <<<JS_CODE

/*
 * Google Places Autocomplete
 */

var geolocation = {$gpa->geolocation};

{$itemCode}
JS_CODE;

				$code = <<<CODE
<style>
{$cssCode}
</style>
<script>
{$jsCode}

function init_autocomplete() {
  {$initAutocomplete}
}

if (navigator.geolocation) {
  navigator.geolocation.getCurrentPosition(function(position) {
    window.browserCoordsAccuracy = position.coords.accuracy;
    window.browserGeolocation = {
      lat: position.coords.latitude,
      lng: position.coords.longitude
    };
    if (typeof window.google !== 'undefined') {
      init_autocomplete();
    }
  });
}
</script>
<script src="//maps.googleapis.com/maps/api/js?key={$gpa->api_key}&libraries=places&callback=init_autocomplete" async defer></script>
</body>
CODE;

				$content = $event->output;
				$event->output =  str_replace("</body>", $code, $content);
			}
		}
	}
}