<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\field_encryption\models;

use app\components\behaviors\DateTrait;
use app\components\behaviors\RelationTrait;
use app\models\Form;
use app\models\User;
use Exception;
use Yii;
use yii\base\Security;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;

/**
 * This is the model class for table "{{%addon_field_encryption}}".
 *
 * @property integer $id
 * @property integer $form_id
 * @property integer $status
 * @property integer $type
 * @property string $key
 * @property integer $permission
 * @property array $users
 * @property integer $created_by
 * @property integer $updated_by
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Form $form
 * @property FieldEncryptionItem[] $items
 * @property User $author
 * @property User $lastEditor
 */
class FieldEncryption extends \yii\db\ActiveRecord
{
    use RelationTrait, DateTrait;

    /**
     * @var int OpenSSL encryption
     */
    const OPENSSL = 1;

    /**
     * @var int gRANT Access
     */
    const DENY = 1;

    /**
     * @var int Grant Access
     */
    const GRANT = 2;

    /**
     * Bank/military grade database storage encryption
     */
    public $cipher = 'AES-256-CBC';

    /**
     * @var Security $security
     */
    protected $security;

    public function init()
    {
        parent::init();

        $this->security = new Security();
        $this->security->cipher = $this->cipher;
    }

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%addon_field_encryption}}';
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['form_id'], 'required'],
            [['form_id'], 'unique', 'message' => Yii::t('app', 'This form already has a configuration.')],
            [['form_id', 'status', 'type', 'permission'], 'integer'],
            [['key'], 'string'],
            ['users', 'required', 'when' => function ($model) {
                return in_array($model->permission, [$this::GRANT, $this::DENY]);
            }, 'whenClient' => "function (attribute, value) {
                return $('#fieldencryption-permission').val() !== '';
            }"],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'form_id' => Yii::t('app', 'Form'),
            'status' => Yii::t('app', 'Status'),
            'type' => Yii::t('app', 'Type'),
            'key' => Yii::t('app', 'Key'),
            'permission' => Yii::t('app', 'Permission to visualize / edit encrypted data'),
            'users' => Yii::t('app', 'Users'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @inheritdoc
     */
    public function afterFind()
    {
        $this->users = explode(',', $this->users);

        parent::afterFind();
    }

    /**
     * @inheritdoc
     */
    public function beforeValidate()
    {
        if (is_array($this->users)) {
            $this->users = implode(',', $this->users);
        }

        return parent::beforeValidate();
    }

    /**
     * Before save the model
     *
     * @param bool $insert
     * @return bool
     * @throws \yii\base\Exception
     */
    public function beforeSave($insert)
    {
        if (!parent::beforeSave($insert)) {
            return false;
        }

        if ($insert) {
            $this->key = base64_encode($this->security->generateRandomKey());
        }

        return true;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForm()
    {
        return $this->hasOne(Form::class, ['id' => 'form_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getItems()
    {
        return $this->hasMany(FieldEncryptionItem::class, ['form_id' => 'form_id']);
    }

    public function setItems($value)
    {
        $this->loadRelated('items', $value);
    }

    /**
     * @inheritdoc
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            // Delete all Items related to this Form
            FieldEncryptionItem::deleteAll(["form_id" => $this->form_id]);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Supported Encryption Types
     */
    public function supportedEncryptionTypes()
    {
        return [
            self::OPENSSL => Yii::t('app', 'OPENSSL'),
        ];
    }

    /**
     * Permission Types
     */
    public function permissions()
    {
        return [
            self::GRANT => Yii::t('app', 'Grant'),
            self::DENY => Yii::t('app', 'Deny'),
        ];
    }

    /**
     * Encrypt data
     *
     * @param $data
     * @return string
     * @throws Exception
     */
    public function encrypt($data)
    {
        return base64_encode($this->security->encryptByKey($data, base64_decode($this->key), $this->created_by));
    }

    /**
     * Decrypt data
     *
     * @param $data
     * @return string
     * @throws Exception
     */
    public function decrypt($data)
    {
        if (is_array($data)) {
            $data = implode(',', $data);
        }
        return $this->security->decryptByKey(base64_decode($data), base64_decode($this->key), $this->created_by);
    }
}