<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\field_encryption;

use app\components\validators\DataValidator;
use app\controllers\AjaxController;
use app\events\SubmissionErrorEvent;
use app\helpers\ArrayHelper;
use app\helpers\Html;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\field_encryption\models\FieldEncryption;
use app\modules\addons\modules\field_encryption\models\FieldEncryptionItem;
use Exception;
use Yii;
use yii\base\ModelEvent;

/**
 * Class Module
 * @package app\modules\addons\modules\field_encryption
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "field_encryption";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'FieldEncryption' => FieldEncryption::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [self::class, 'onFormDeleted']
                ]
            ],
            DataValidator::class => [
                'beforeUniqueFieldsValidation' => [
                    [self::class, 'onBeforeUniqueFieldsValidation']
                ]
            ],
            FormSubmission::class => [
                'afterFind' => [
                    [self::class, 'onAfterFindFormSubmission']
                ],
                'beforeInsert' => [
                    [self::class, 'onBeforeInsertFormSubmission']
                ],
                'beforeUpdate' => [
                    [self::class, 'onBeforeUpdateFormSubmission']
                ],
                'afterUpdate' => [
                    [self::class, 'onAfterUpdateFormSubmission']
                ],
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = FieldEncryption::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new FieldEncryption();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new FieldEncryptionItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event, $event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            FieldEncryption::deleteAll(['form_id' => $event->sender->id]);
        }
    }

    /**
     * Event Handler
     * Before Unique Fields Validation
     *
     * @param $event
     * @throws Exception
     */
    public static function onBeforeUniqueFieldsValidation($event)
    {
        if (isset($event, $event->sender, $event->sender->data, $event->sender->submissionModel, $event->sender->submissionModel->form_id)) {
            $model = FieldEncryption::findOne(['form_id' => $event->sender->submissionModel->form_id, 'status' => 1]);

            if (isset($model, $model->form_id)) {

                // Data Validator
                $dataValidator = $event->sender;

                // Convert submission data (json) into an array
                /** @var array $data */
                $data = is_string($dataValidator->data) ? json_decode($dataValidator->data, true) : $dataValidator->data;

                $uniqueFields = [];
                foreach ($dataValidator->uniqueFields as $uniqueField) {
                    if (isset($uniqueField['name'])) {
                        // Get label
                        $label = Yii::t('app', 'the input value');
                        if (isset($uniqueField["label"])) {
                            $label = $uniqueField["label"];
                        } elseif (isset($uniqueField["data-label"])) {
                            $label = $uniqueField["data-label"];
                        }
                        $uniqueFields[$uniqueField['name']] = $label;
                    }
                }

                $encryptedFields = ArrayHelper::getColumn($model->items, 'field_id');
                $uniqueEncryptedFields = array_intersect(array_keys($uniqueFields), $encryptedFields);

                if (!empty($uniqueEncryptedFields)) {
                    /** @var FormSubmission[] $submissions */
                    $submissions = FormSubmission::find()
                        ->select(['id', 'data'])
                        ->where('form_id=:form_id', [':form_id' => $model->form_id])
                        ->all();

                    foreach ($submissions as $submission) {
                        if (!empty($submission->data)) {
                            foreach ($uniqueEncryptedFields as $field) {
                                if (!empty($submission->data[$field])) {
                                    $value = $model->decrypt($submission->data[$field]);
                                    if (isset($data[$field]) && trim($data[$field]) === $value) {
                                        // Only when we are not updating the form submission
                                        if ($dataValidator->submissionModel->id !== $submission->id) {
                                            $errorEvent = new SubmissionErrorEvent();
                                            $errorEvent->message = Yii::t('app', '{attribute} "{value}" has already been taken.');
                                            $errorEvent->fieldName = $field;
                                            $event->sender->trigger(SubmissionErrorEvent::EVENT_NAME, $errorEvent);
                                            $event->sender->addError($field, $uniqueFields[$field], $value, $errorEvent->message);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Event Handler
     * Executed after find a FormSubmission model
     *
     * @param $event
     * @throws Exception
     */
    public static function onAfterFindFormSubmission($event)
    {
        if (isset($event, $event->sender, $event->sender->id, $event->sender->form) && $event->sender instanceof FormSubmission) {

            $model = FieldEncryption::findOne(['form_id' => $event->sender->form->id, 'status' => 1]);

            if (isset($model, $model->form_id)) {

                // Convert submission data (json) into an array
                /** @var array $data */
                $data = is_string($event->sender->data) ? json_decode($event->sender->data, true) : $event->sender->data;

                foreach ($model->items as $item) {
                    if (isset($item->field_id, $data[$item->field_id])) {
                        $value = $data[$item->field_id];
                        // Current user is the configuration owner
                        $condition1 = Yii::$app->user->id === $model->created_by;
                        // Current user has a granted permission
                        $condition2 = $model->permission === $model::GRANT && in_array(Yii::$app->user->id, $model->users);
                        // Current user has not a denied permission
                        $condition3 = $model->permission === $model::DENY && !in_array(Yii::$app->user->id, $model->users);
                        // Current user is an anonymous user
                        $condition4 = Yii::$app->user->isGuest;
                        if ($condition1 || $condition2 || $condition3 || $condition4) {
                            // Decrypted Field Value
                            $data[$item->field_id] = $model->decrypt($value);
                            // Check if field has collected an array
                            $componentInfo = explode("_", $item->field_id);
                            $componentType = $componentInfo[0];
                            if ($componentType === "checkbox" || $componentType === "selectlist") {
                                // For Checkboxes and Select List Multiple
                                $data[$item->field_id] = explode(',', $data[$item->field_id]);
                            }
                        } else {
                            // Display alternative text
                            $data[$item->field_id] = Html::encode($item->display);
                        }
                    }
                }

                $event->sender->data = $data;
            }
        }
    }

    /**
     * Event Handler
     * Executed before a FormSubmission model is inserted
     *
     * @param ModelEvent $event
     * @throws Exception
     */
    public static function onBeforeInsertFormSubmission($event)
    {
        if (isset($event, $event->sender, $event->isValid) && $event->isValid && $event->sender instanceof FormSubmission) {

            $model = FieldEncryption::findOne(['form_id' => $event->sender->form->id, 'status' => 1]);

            if (isset($model, $model->form_id)) {

                // Convert submission data (json) into an array
                /** @var array $data */
                $data = is_string($event->sender->data) ? json_decode($event->sender->data, true) : $event->sender->data;

                foreach ($model->items as $item) {
                    if (isset($item->field_id, $data[$item->field_id])) {
                        $value = $data[$item->field_id];
                        if (is_array($value)) { // For Checkboxes and Select List Multiple
                            $value = implode(',', $value);
                        }
                        // Insert Encrypted Field Value
                        $data[$item->field_id] = $model->encrypt($value);
                    }
                }

                $event->sender->data = $data;
            }
        }
    }

    /**
     * Event Handler
     * Executed before a FormSubmission model is updated
     *
     * @param ModelEvent $event
     * @throws Exception
     */
    public static function onBeforeUpdateFormSubmission($event)
    {
        if (isset($event, $event->sender, $event->isValid) && $event->isValid && $event->sender instanceof FormSubmission) {

            $model = FieldEncryption::findOne(['form_id' => $event->sender->form->id, 'status' => 1]);

            if (isset($model, $model->form_id)) {

                // Convert submission data (json) into an array
                /** @var array $data */
                $data = is_string($event->sender->data) ? json_decode($event->sender->data, true) : $event->sender->data;
                $oldData = is_string($event->sender->oldAttributes['data']) ? json_decode($event->sender->oldAttributes['data'], true) : $event->sender->oldAttributes['data'];

                foreach ($model->items as $item) {
                    if (isset($item->field_id, $data[$item->field_id])) {
                        // Current user is the configuration owner
                        $condition1 = Yii::$app->user->id === $model->created_by;
                        // Current user has a granted permission
                        $condition2 = $model->permission === $model::GRANT && in_array(Yii::$app->user->id, $model->users);
                        // Current user has not a denied permission
                        $condition3 = $model->permission === $model::DENY && !in_array(Yii::$app->user->id, $model->users);
                        if ($condition1 || $condition2 || $condition3) {
                            // Update Field Value
                            $value = $data[$item->field_id];
                            if (is_array($value)) { // For Checkboxes and Select List Multiple
                                $value = implode(',', $value);
                            }
                            $data[$item->field_id] = $model->encrypt($value);
                        } else {
                            // Restore previous field value
                            $data[$item->field_id] = $oldData[$item->field_id];
                        }
                    }
                }

                $event->sender->data = $data;
            }
        }
    }

    /**
     * Event Handler
     * Executed after a FormSubmission model is updated
     *
     * @param ModelEvent $event
     * @throws Exception
     */
    public static function onAfterUpdateFormSubmission($event)
    {
        if (isset($event, $event->sender) && $event->sender instanceof FormSubmission) {

            $model = FieldEncryption::findOne(['form_id' => $event->sender->form->id, 'status' => 1]);

            if (isset($model, $model->form_id)) {

                // Convert submission data (json) into an array
                /** @var array $data */
                $data = is_string($event->sender->data) ? json_decode($event->sender->data, true) : $event->sender->data;

                foreach ($model->items as $item) {
                    if (isset($item->field_id, $data[$item->field_id])) {
                        $value = $data[$item->field_id];
                        // Current user is the configuration owner
                        $condition1 = Yii::$app->user->id === $model->created_by;
                        // Current user has a granted permission
                        $condition2 = $model->permission === $model::GRANT && in_array(Yii::$app->user->id, $model->users);
                        // Current user has not a denied permission
                        $condition3 = $model->permission === $model::DENY && !in_array(Yii::$app->user->id, $model->users);
                        if ($condition1 || $condition2 || $condition3) {
                            // Decrypted Field Value
                            $data[$item->field_id] = $model->decrypt($value);
                            // Check if field has collected an array
                            $componentInfo = explode("_", $item->field_id);
                            $componentType = $componentInfo[0];
                            if ($componentType === "checkbox" || $componentType === "selectlist") {
                                // For Checkboxes and Select List Multiple
                                $data[$item->field_id] = explode(',', $data[$item->field_id]);
                            }
                        } else {
                            // Display alternative text
                            $data[$item->field_id] = Html::encode($item->display);
                        }
                    }
                }

                $event->sender->data = $data;
            }
        }
    }
}