<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\eu_vat;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\data_validation\models\DataValidation;
use app\modules\addons\modules\eu_vat\models\EuVat;
use app\modules\addons\modules\eu_vat\models\EuVatItem;
use app\modules\addons\modules\eu_vat\services\EuVatService;
use Exception;
use Yii;
use yii\helpers\Url;
use yii\httpclient\Client;
use yii\web\Response;
use yii\web\View;

class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "eu_vat";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'EuVat' => EuVat::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
            FormSubmission::class => [
                'afterValidate' => [
                    [Module::class, 'onFormSubmissionAfterValidate']
                ],
            ],
            View::class => [
                'afterRender' => [
                    [Module::class, 'onViewAfterRender']
                ],
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = EuVat::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new EuVat();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new EuVatItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->eu_vat_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     * @throws \Throwable
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = EuVat::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a Form Submission model is validated
     *
     * @param $event
     */
    public static function onFormSubmissionAfterValidate($event)
    {
        if (isset($event, $event->sender, $event->sender->form_id) && $event->sender instanceof FormSubmission) {
            $model = EuVat::findOne(['form_id' => $event->sender->form_id, 'status' => EuVat::ON]);

            if (isset($model, $model->form_id)) {

                $submissionData = $event->sender->data;

                foreach ($model->items as $item) {
                    // Skip On Empty
                    if (isset($item->vat_number) && !empty($submissionData[$item->vat_number])) {

                        $value = $submissionData[$item->vat_number];
                        $value = is_array($value) ? implode(',', $value) : $value;

                        try {
                            $service = new EuVatService();
                            $isValidVATNumber = $service->isValidVATNumber($value);
                        } catch (\Exception $e) {
                            $isValidVATNumber = false;
                            Yii::error($e);
                        }

                        if (!$isValidVATNumber) {
                            self::showErrorMessage($item, Yii::t('app', 'Please enter a valid VAT Number.'));
                        }
                    }
                }
            }
        }
    }

    /**
     * Event Handler
     * After a View is rendered
     *
     * @param $event
     * @throws \Exception
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "app/embed" ) {

            $formModel = $event->sender->context->getFormModel();
            $model = EuVat::findOne(['form_id' => $formModel->id, 'status' => EuVat::ON]);

            if ($model && !empty($model->items)) {

                $service = new EuVatService();

                if ($model->business_country_code) {
                    $service->setBusinessCountryCode($model->business_country_code);
                }

                $countryCode = $service->getIPBasedCountry();

                $loadingTxt = Yii::t('app', 'Loading...');
                $url = Url::to(['/addons/eu_vat/check/tax-rate', 'id' => $model->id]);
                $code = '';
                $ipBasedCountryCode = '';

                foreach ($model->items as $item) {
                    $countryCodeField = !empty($item->country_code) ? "#" . $item->country_code : null;
                    $vatTaxRateField = !empty($item->vat_tax_rate) ? "#" . $item->vat_tax_rate : null;
                    $netAmountField = !empty($item->vat_subtotal) ? "#" . $item->vat_subtotal : null;
                    $taxesField = !empty($item->vat_taxes) ? "#" . $item->vat_taxes : null;
                    $grossAmountField = !empty($item->vat_total) ? "#" . $item->vat_total : null;
                    $vatNumberField = !empty($item->vat_number) ? "#" . $item->vat_number : null;

                    $selector = $countryCodeField ? $countryCodeField : null;
                    $selector = $vatNumberField ? $selector . ', ' . $vatNumberField : $selector;
                    $selector = $taxesField ? $selector . ', ' . $taxesField : $selector;
                    $selector = $netAmountField ? $selector . ', ' . $netAmountField : $selector;

                    if ($model->ip_based_country && $countryCode) {
                        $ipBasedCountryCode = <<<JS
$("{$countryCodeField}").val("{$countryCode}").trigger("change");
JS;
                    }


                    $code .= <<<CODE

<script type="text/javascript">
$(document).ready(function() {
    var submitBtn = $(":submit");
    var submitTxt = [];
    submitBtn.each(function (i) {
        submitTxt[i] = $(this).html();
    });
    var euVatDelay = function (callback, ms) {
        var timer = 0;
        return function() {
            var context = this, args = arguments;
            clearTimeout(timer);
            timer = setTimeout(function () {
                callback.apply(context, args);
            }, ms || 0);
        };
    }
    var euVatCalculator = function(e) {
        var countryCode = $("{$countryCodeField}").val();
        var vatNumber = $("{$vatNumberField}").val();
        var netAmount = $("{$netAmountField}").val();
        submitBtn.html("{$loadingTxt}");
        $.post( "{$url}", { country_code: countryCode, vat_number: vatNumber, net_amount: netAmount })
            .done(function(data) {
                if (typeof data.taxes !== 'undefined' ) {
                    $("{$taxesField}").val(data.taxes);
                }
                if (typeof data.gross_amount !== 'undefined' ) {
                    $("{$grossAmountField}").val(data.gross_amount);
                }
                if (typeof data.tax_rate !== 'undefined' ) {
                    $("{$vatTaxRateField}").val(data.tax_rate).trigger('change').trigger('input');
                }
                submitBtn.each(function (i) {
                    $(this).html(submitTxt[i]);
                });
            });
    };
    $("form").find("{$selector}").on('keyup change', euVatDelay(function () {
      euVatCalculator();
    }, 500));
    {$ipBasedCountryCode}
});
</script>
CODE;
                }

                $code .= '
</body>';

                $content = $event->output;
                $event->output =  str_replace("</body>", $code, $content);
            }
        }
    }

    /**
     * Show Error Message
     *
     * @param EuVatItem $item
     * @param string $message
     */
    public static function showErrorMessage($item, $message)
    {
        $errors = [
            'field' => $item->vat_number,
            'messages' => [$message],
        ];
        $response = Yii::$app->getResponse();
        $response->format = Response::FORMAT_JSON;
        $response->data = array(
            'action'  => 'submit',
            'success' => false,
            'id' => 0,
            'message' => Yii::t('app', 'There is {startTag}an error in your submission{endTag}.', [
                'startTag' => '<strong>',
                'endTag' => '</strong>',
            ]),
            'errors' => [$errors],
        );
        $response->send();
        exit;
    }
}
