<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\data_validation\controllers;

use app\components\User;
use app\models\Form;
use app\modules\addons\modules\data_validation\models\DataValidation;
use app\modules\addons\modules\data_validation\models\DataValidationItem;
use app\modules\addons\modules\data_validation\models\DataValidationSearch;
use kartik\depdrop\DepDropAction;
use SimpleExcel\SimpleExcel;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\web\UploadedFile;

/**
 * AdminController implements the CRUD actions for DataValidation model.
 */
class AdminController extends Controller
{

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'ways'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('DataValidation')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = DataValidation::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = \app\helpers\ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\data_validation\models\DataValidation',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all DataValidation models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new DataValidationSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single model.
	 * @param $id
	 *
	 * @return string
	 * @throws NotFoundHttpException
	 */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new DataValidation();

        if ($model->load(Yii::$app->request->post()) && $model->validate()) {
            $items = Yii::$app->request->post('DataValidationItem',[]);
            foreach ($items as $i => $item) {
                $file = UploadedFile::getInstanceByName('DataValidationItem['.$i.'][field_values]');
                if (isset($file->tempName)) {
                    $fp = fopen($file->tempName, 'r');
                    $str = fread($fp, filesize($file->tempName));
                    fclose($fp);
                    $excel = new SimpleExcel('CSV');
                    $excel->parser->loadString($str);
                    $items[$i]['field_values'] = json_encode($excel->parser->getColumn(1), true);
                } elseif (isset($items[$i]['field_values']) && !empty($items[$i]['field_values'])) {
                    $values = str_replace(["\t", "\r"], '', $items[$i]['field_values']);
                    $values = explode("\n", $values);
                    $values = array_map('trim', $values);
                    $items[$i]['field_values'] = json_encode($values, true);
                } else {
                    $items[$i]['field_values'] = null;
                }
            }
            $model->items = $items;
            $model->save();
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The form has been successfully configured.')
            );
            return $this->redirect(['index']);
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = \app\helpers\ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'itemModel' => new DataValidationItem,
            'forms' => $forms,
        ]);
    }

    /**
     * Updates an existing model.
     * If update is successful, the browser will be redirected to the 'view' page.
	 * @param $id
	 *
	 * @return string|\yii\web\Response
	 * @throws NotFoundHttpException
	 */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);

        if ($model->load(Yii::$app->request->post()) && $model->validate()) {
            $items = Yii::$app->request->post('DataValidationItem',[]);
            foreach ($items as $i => $item) {
                $file = UploadedFile::getInstanceByName('DataValidationItem['.$i.'][field_values]');
                if (isset($file->tempName)) {
                    $fp = fopen($file->tempName, 'r');
                    $str = fread($fp, filesize($file->tempName));
                    fclose($fp);
                    $excel = new SimpleExcel('CSV');
                    $excel->parser->loadString($str);
                    $items[$i]['field_values'] = json_encode($excel->parser->getColumn(1), true);
                } elseif (isset($items[$i]['field_values']) && !empty($items[$i]['field_values'])) {
                    $values = str_replace(["\t", "\r"], '', $items[$i]['field_values']);
                    $values = explode("\n", $values);
                    $values = array_map('trim', $values);
                    $items[$i]['field_values'] = json_encode($values, true);
                } else {
                    if (isset($items[$i]['field_id'])) {
                        $fieldID = $items[$i]['field_id'];
                        /** @var DataValidationItem $item */
                        $item = DataValidationItem::find()->where(['field_id' => $fieldID])->one();
                        if ($item) {
                            $items[$i]['field_values'] = $item->field_values;
                        }
                    }
                }
            }
            $model->items = $items;
            $model->save();
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The form has been successfully updated.')
            );
            return $this->redirect(['index']);
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = \app\helpers\ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'itemModel' => new DataValidationItem,
            'forms' => $forms,
        ]);
    }

    /**
     * Enable / Disable multiple Forms
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {

        $models = DataValidation::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
	 * @param $id
	 *
	 * @return \yii\web\Response
	 * @throws NotFoundHttpException
	 * @throws \Exception
	 * @throws \yii\db\StaleObjectException
	 */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();
        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The form configuration has been successfully deleted.')
        );
        return $this->redirect(['index']);
    }

    /**
     * Finds the model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return DataValidation the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = DataValidation::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}