<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\data_validation;

use app\controllers\AjaxController;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\data_validation\models\DataValidation;
use app\modules\addons\modules\data_validation\models\DataValidationItem;
use Yii;
use yii\base\BaseObject;
use yii\helpers\StringHelper;
use yii\validators\CompareValidator;
use yii\validators\RangeValidator;
use yii\validators\StringValidator;
use yii\web\Response;

/**
 * Class Module
 * @package app\modules\addons\modules\data_validation
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "data_validation";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';
    public static $errors = [];

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'DataValidation' => DataValidation::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            FormSubmission::class => [
                'afterValidate' => [
                    [Module::class, 'onFormSubmissionAfterValidate']
                ],
            ],
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormBeforeDelete']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = DataValidation::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new DataValidation();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new DataValidationItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a Form model is deleted
     *
	 * @param $event
	 *
     * @throws \Exception|\Throwable in case delete failed.
	 */
    public static function onFormBeforeDelete($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $model = DataValidation::findOne(['form_id' => $event->sender->id]);
            if ($model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a Form Submission model is validated
     *
     * @param $event
     */
    public static function onFormSubmissionAfterValidate($event)
    {
        if (isset($event, $event->sender, $event->sender->form_id) && $event->sender instanceof FormSubmission) {
            $model = DataValidation::findOne(['form_id' => $event->sender->form_id, 'status' => DataValidation::ON]);

            if (isset($model, $model->form_id)) {
                $submissionData = $event->sender->data;
                self::$errors = $event->sender->getErrors();
                foreach ($model->items as $item) {
                    // Skip On Empty
                    if (isset($item->field_id) && !empty($submissionData[$item->field_id])) {

                        $value = $submissionData[$item->field_id];
                        $value = is_array($value) ? implode(',', $value) : $value;

                        self::validate($value, $item, $submissionData);
                    }
                }
                $event->sender->clearErrors();
                $event->sender->addErrors(self::$errors);
            }
        }
    }

    /**
     * Validate
     *
     * @param string $value
     * @param DataValidationItem $item
     * @param array $submissionData
     */
    public static function validate($value, $item, $submissionData)
    {
        if (is_array($value)) {
            // For Checkboxes and Select List
            foreach ($value as $val) {
                self::validateValue($val, $item, $submissionData);
            }
        } else {
            // For other fields
            self::validateValue($value, $item, $submissionData);
        }
    }

    /**
     * Validate Value
     *
     * @param string $value
     * @param DataValidationItem $item
     * @param array $submissionData
     */
    protected static function validateValue($value, $item, $submissionData)
    {

        switch ($item->validation_rule) {
            case DataValidationItem::RULE_IS:
                if (!empty($item->field_values)) {
                    self::range($value, $item, [
                        'range' => json_decode($item->field_values, true),
                    ]);
                }
                if (!empty($submissionData[$item->validation_field])) {
                    self::compare($value, $item, [
                        'compareValue' => $submissionData[$item->validation_field],
                        'operator' => '=='
                    ]);
                }
                break;
            case DataValidationItem::RULE_IS_NOT:
                if (!empty($item->field_values)) {
                    self::range($value, $item, [
                        'range' => json_decode($item->field_values, true),
                        'not' => true,
                    ]);
                }
                if (!empty($submissionData[$item->validation_field])) {
                    self::compare($value, $item, [
                        'compareValue' => $submissionData[$item->validation_field],
                        'operator' => '!='
                    ]);
                }
                break;
            case DataValidationItem::RULE_CONTAINS:
                if (!empty($item->field_values)) {
                    self::string($value, $item, [
                        'operator' => 'contains',
                        'containsValue' => json_decode($item->field_values, true),
                    ]);
                }
                if (!empty($submissionData[$item->validation_field])) {
                    self::string($value, $item, [
                        'operator' => 'contains',
                        'containsValue' => $submissionData[$item->validation_field],
                    ]);
                }
                break;
            case DataValidationItem::RULE_DOES_NOT_CONTAIN:
                if (!empty($item->field_values)) {
                    self::string($value, $item, [
                        'operator' => 'doesNotContains',
                        'doesNotContainsValue' => json_decode($item->field_values, true),
                    ]);
                }
                if (!empty($submissionData[$item->validation_field])) {
                    self::string($value, $item, [
                        'operator' => 'doesNotContains',
                        'doesNotContainsValue' => $submissionData[$item->validation_field],
                    ]);
                }
                break;
            case DataValidationItem::RULE_STARTS_WITH:
                if (!empty($item->field_values)) {
                    self::string($value, $item, [
                        'operator' => 'startsWith',
                        'with' => json_decode($item->field_values, true),
                    ]);
                }
                if (!empty($submissionData[$item->validation_field])) {
                    self::compare($value, $item, [
                        'operator' => 'startsWith',
                        'with' => $submissionData[$item->validation_field],
                    ]);
                }
                break;
            case DataValidationItem::RULE_ENDS_WITH:
                if (!empty($item->field_values)) {
                    self::string($value, $item, [
                        'operator' => 'endsWith',
                        'with' => json_decode($item->field_values, true),
                    ]);
                }
                if (!empty($submissionData[$item->validation_field])) {
                    self::compare($value, $item, [
                        'operator' => 'endsWith',
                        'with' => $submissionData[$item->validation_field],
                    ]);
                }
                break;
        }
    }

    /**
     * String Validator
     *
     * @param $value
     * @param $item
     * @param array $options
     */
    public static function string($value, $item, $options = [])
    {
        if (isset($options['operator'], $options['containsValue']) && $options['operator'] === 'contains') {

            $count = 0;

            if (is_array($options['containsValue'])) {
                foreach ($options['containsValue'] as $containsValue) {
                    if (is_array($value)) {
                        if (in_array($containsValue, $value))  {
                            $count++;
                        }
                    } else {
                        if (self::match($containsValue, $value)) {
                            $count++;
                        }
                    }
                }
            } else {
                if (self::match($options['containsValue'], $value)) {
                    $count++;
                }
            }

            if ($count === 0) {

                $message = Yii::t('app', 'the input value should contains specific values');

                if (!empty($item->field_message)) {
                    $message = $item->field_message;
                }

                self::addErrorMessage($item, $message);
            }

        } elseif (isset($options['operator'], $options['doesNotContainsValue']) && $options['operator'] === 'doesNotContains') {

            $count = 0;

            if (is_array($options['doesNotContainsValue'])) {
                foreach ($options['doesNotContainsValue'] as $doesNotContainsValue) {
                    if (is_array($value)) {
                        if (in_array($doesNotContainsValue, $value))  {
                            $count++;
                        }
                    } else {
                        if (self::match($doesNotContainsValue, $value)) {
                            $count++;
                        }
                    }
                }
            } else {
                if (self::match($options['doesNotContainsValue'], $value)) {
                    $count++;
                }
            }

            if ($count > 0) {
                $message = Yii::t('app', 'the input value should not contains specific values');

                if (!empty($item->field_message)) {
                    $message = $item->field_message;
                }

                self::addErrorMessage($item, $message);
            }

        } elseif (isset($options['operator']) && $options['operator'] === 'startsWith') {

            if (is_array($options['with'])) {
                foreach ($options['with'] as $with) {
                    if (StringHelper::startsWith($value, $with))  {
                        return;
                    }
                }
            } else {
                if (StringHelper::startsWith($value, $options['with']))  {
                    return;
                }
            }

            $message = Yii::t('app', 'the input value needs to starts with specific values');

            if (!empty($item->field_message)) {
                $message = $item->field_message;
            }

            self::addErrorMessage($item, $message);

        } elseif (isset($options['operator']) && $options['operator'] === 'endsWith') {

            if (is_array($options['with'])) {
                foreach ($options['with'] as $with) {
                    if (StringHelper::endsWith($value, $with))  {
                        return;
                    }
                }
            } else {
                if (StringHelper::endsWith($value, $options['with']))  {
                    return;
                }
            }

            $message = Yii::t('app', 'the input value needs to ends with specific values');

            if (!empty($item->field_message)) {
                $message = $item->field_message;
            }

            self::addErrorMessage($item, $message);
        }
    }

    /**
     * Compare Validator
     *
     * @param $value
     * @param $item
     * @param array $options
     */
    public static function compare($value, $item, $options = [])
    {
        if (!empty($item->field_message)) {
            $options['message'] = $item->field_message;
        }
        $validator = new CompareValidator($options);
        if (!$validator->validate($value, $message)) {
            self::addErrorMessage($item, $message);
        }
    }

    /**
     * Range Validator
     *
     * @param $value
     * @param $item
     * @param array $options
     */
    public static function range($value, $item, $options = [])
    {
        if (!empty($item->field_message)) {
            $options['message'] = $item->field_message;
        }
        $validator = new RangeValidator($options);
        if (!$validator->validate($value, $message)) {
            self::addErrorMessage($item, $message);
        }
    }

    /**
     * Add Error Message
     *
     * @param DataValidationItem $item
     * @param string $message
     */
    public static function addErrorMessage($item, $message)
    {
        $errors = self::$errors;
        $errors[$item->field_id] = [$message];
        self::$errors = $errors;
    }

    /**
     * Show Error Messages
     */
    public static function showErrorMessages()
    {
        $response = Yii::$app->getResponse();
        $response->format = Response::FORMAT_JSON;
        $response->data = array(
            'action'  => 'submit',
            'success' => false,
            'id' => 0,
            'message' => Yii::t('app', 'There is {startTag}an error in your submission{endTag}.', [
                'startTag' => '<strong>',
                'endTag' => '</strong>',
            ]),
            'errors' => [self::$errors],
        );
        $response->send();
        exit;
    }

    public static function match($value, $string)
    {
        return preg_match("/\b$value\b/i", $string);
    }
}