<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\custom_error_message\models;

use app\components\behaviors\DateTrait;
use app\models\User;
use Yii;
use app\models\Form;
use app\components\behaviors\RelationTrait;
use yii\behaviors\BlameableBehavior;
use yii\behaviors\TimestampBehavior;

/**
 * This is the model class for table "{{%addon_custom_error_message}}".
 *
 * @property integer $id
 * @property integer $form_id
 * @property integer $status
 * @property integer $created_by
 * @property integer $updated_by
 * @property integer $created_at
 * @property integer $updated_at
 *
 * @property Form $form
 * @property CustomErrorMessageItem[] $items
 * @property User $author
 * @property User $lastEditor
 */
class CustomErrorMessage extends \yii\db\ActiveRecord
{
    use RelationTrait, DateTrait;

    public $validationTypes = [];

    public $validationMessages = [];

    /**
     * @inheritdoc
     */
    public function init()
    {
        /**
         * Validation Types
         */
        $this->validationTypes = [
            'REQUIRED' => Yii::t('app', 'Required'),
            'UNIQUE' => Yii::t('app', 'Unique'),
            'INVALID_EMAIL' => Yii::t('app', 'Invalid Email'),
            'INVALID_REGEX' => Yii::t('app', 'Invalid Pattern'),
            'INVALID_URL' => Yii::t('app', 'Invalid Url'),
            'INVALID_DATE' => Yii::t('app', 'Invalid Date'),
            'TOO_SMALL_DATE' => Yii::t('app', 'Too Small Date'),
            'TOO_BIG_DATE' => Yii::t('app', 'Too Big Date'),
            'INVALID_STRING' => Yii::t('app', 'Invalid String'),
            'TOO_SHORT_STRING' => Yii::t('app', 'Too Short String'),
            'TOO_LONG_STRING' => Yii::t('app', 'Too Long String'),
            'INVALID_INTEGER' => Yii::t('app', 'Invalid Integer'),
            'INVALID_NUMBER' => Yii::t('app', 'Invalid Number'),
            'TOO_SMALL_NUMBER' => Yii::t('app', 'Too Small Number'),
            'TOO_BIG_NUMBER' => Yii::t('app', 'Too Big Number'),
            'INVALID_FILE' => Yii::t('app', 'Invalid File'),
            'REQUIRED_FILE' => Yii::t('app', 'Required File'),
            'INVALID_FILE_EXTENSIONS' => Yii::t('app', 'Invalid File Extensions'),
            'TOO_SMALL_FILE' => Yii::t('app', 'Too Small File'),
            'TOO_BIG_FILE' => Yii::t('app', 'Too Big File'),
            'TOO_FEW_FILES' => Yii::t('app', 'Too Few Files'),
            'TOO_MANY_FILES' => Yii::t('app', 'Too Many Files'),
            'INVALID_PHONE' => Yii::t('app', 'Invalid Phone'),
            'REQUIRED_RECAPTCHA' => Yii::t('app', 'Required reCAPTCHA'),
            'INVALID_RECAPTCHA' => Yii::t('app', 'Invalid reCAPTCHA'),
        ];

        /**
         * Validation Messages
         */
        $this->validationMessages = [
            'REQUIRED' => Yii::t('app', '{attribute} cannot be blank.'),
            'UNIQUE' => Yii::t('app', '{attribute} "{value}" has already been taken.'),
            'INVALID_EMAIL' => Yii::t('app', '{attribute} is not a valid email address.'),
            'INVALID_REGEX' => Yii::t('app', '{attribute} is invalid.'),
            'INVALID_URL' => Yii::t('app', '{attribute} is not a valid URL.'),
            'INVALID_DATE' => Yii::t('app', 'The format of {attribute} is invalid.'),
            'TOO_SMALL_DATE' => Yii::t('app', '{attribute} must be no less than {min}.'),
            'TOO_BIG_DATE' => Yii::t('app', '{attribute} must be no greater than {max}.'),
            'INVALID_STRING' => Yii::t('app', '{attribute} must be a string.'),
            'TOO_SHORT_STRING' => Yii::t('app', '{attribute} should contain at least {min, number} {min, plural, one{character} other{characters}}.'),
            'TOO_LONG_STRING' => Yii::t('app', '{attribute} should contain at most {max, number} {max, plural, one{character} other{characters}}.'),
            'INVALID_INTEGER' => Yii::t('app', '{attribute} must be an integer.'),
            'INVALID_NUMBER' => Yii::t('app', '{attribute} must be a number.'),
            'TOO_SMALL_NUMBER' => Yii::t('app', '{attribute} must be no less than {min}.'),
            'TOO_BIG_NUMBER' => Yii::t('app', '{attribute} must be no greater than {max}.'),
            'INVALID_FILE' => Yii::t('app', 'File upload failed.'),
            'REQUIRED_FILE' => Yii::t('app', 'Please upload a file.'),
            'INVALID_FILE_EXTENSIONS' => Yii::t('app', 'Only files with these extensions are allowed: {extensions}.'),
            'TOO_SMALL_FILE' => Yii::t('app', 'The file "{file}" is too small. Its size cannot be smaller than {formattedLimit}.'),
            'TOO_BIG_FILE' => Yii::t('app', 'The file "{file}" is too big. Its size cannot exceed {formattedLimit}.'),
            'TOO_FEW_FILES' => Yii::t('app', 'You should upload at least {limit, number} {limit, plural, one{file} other{files}}.'),
            'TOO_MANY_FILES' => Yii::t('app', 'You can upload at most {limit, number} {limit, plural, one{file} other{files}}.'),
            'INVALID_PHONE' => Yii::t('app', '{attribute} is not a valid phone number.'),
            'REQUIRED_RECAPTCHA' => Yii::t('app', 'The captcha is a required field.'),
            'INVALID_RECAPTCHA' => Yii::t('app', 'The captcha code you entered was incorrect.'),
        ];

        parent::init();
    }

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%addon_custom_error_message}}';
    }

    /**
     * @inheritdoc
     */
    public static function primaryKey()
    {
        return ['id'];
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            BlameableBehavior::class,
            TimestampBehavior::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['form_id'], 'required'],
            [['form_id'], 'unique', 'message' => Yii::t('app', 'This form already has a configuration.')],
            [['form_id', 'status'], 'integer'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('app', 'ID'),
            'form_id' => Yii::t('app', 'Form'),
            'status' => Yii::t('app', 'Status'),
            'created_by' => Yii::t('app', 'Created By'),
            'updated_by' => Yii::t('app', 'Updated By'),
            'created_at' => Yii::t('app', 'Created At'),
            'updated_at' => Yii::t('app', 'Updated At'),
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getForm()
    {
        return $this->hasOne(Form::class, ['id' => 'form_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuthor()
    {
        return $this->hasOne(User::class, ['id' => 'created_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getLastEditor()
    {
        return $this->hasOne(User::class, ['id' => 'updated_by']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getItems()
    {
        return $this->hasMany(CustomErrorMessageItem::class, ['form_id' => 'form_id']);
    }

    public function setItems($value)
    {
        $this->loadRelated('items', $value);
    }

    /**
     * @inheritdoc
     */
    public function beforeDelete()
    {
        if (parent::beforeDelete()) {
            // Delete all Items related to this Form
            CustomErrorMessageItem::deleteAll(["form_id" => $this->form_id]);
            return true;
        } else {
            return false;
        }
    }
}